// scriplib.c

// 
// This file was originally part of the Quake2 source release.  
// I've modified the script library code to malloc of a shared
// memory arena.
//
// Rick
//

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "pfbsp_int.h"
#include "scriptlib.h"
#include "qfiles.h"

#define true 1
#define false 0

/*
=============================================================================

						PARSING STUFF

=============================================================================
*/

typedef struct
{
  char	filename[1024];
  char    *buffer,*script_p,*end_p;
  int     line;
} script_t;

#define	MAX_INCLUDES	8
script_t	*scriptstack = NULL;
script_t	*script;
int			scriptline;

char    token[MAXTOKEN];
int endofscript;
int tokenready;                     // only true if UnGetToken was just called


/*
==============
ParseFromMemory
==============
*/
void ParseFromMemory (char *buffer, int size)
{
  if (scriptstack == NULL) 
    scriptstack = (script_t *)pfMalloc(sizeof(script_t)*MAX_INCLUDES, 
				       pfGetSharedArena());
  script = scriptstack;
  script++;
  if (script == &scriptstack[MAX_INCLUDES])
    fprintf(stderr, "script file exceeded MAX_INCLUDES");
  strcpy (script->filename, "memory buffer" );
  
  script->buffer = buffer;
  script->line = 1;
  script->script_p = script->buffer;
  script->end_p = script->buffer + size;
  
  endofscript = false;
  tokenready = false;
}


/*
==============
UnGetToken

Signals that the current token was not used, and should be reported
for the next GetToken.  Note that

GetToken (true);
UnGetToken ();
GetToken (false);

could cross a line boundary.
==============
*/
void UnGetToken (void)
{
  tokenready = true;
}


int EndOfScript (int crossline)
{
  if (!crossline)
    fprintf (stderr, "Line %i is incomplete\n",scriptline);
  
  if (!strcmp (script->filename, "memory buffer"))
    {
      endofscript = true;
      return false;
    }
  
  pfFree(script->buffer);
  if (script == scriptstack+1)
    {
      endofscript = true;
      return false;
    }
  script--;
  scriptline = script->line;
  printf ("returning to %s\n", script->filename);
  return GetToken (crossline);
}

/*
==============
GetToken
==============
*/
int GetToken (int crossline)
{
  char    *token_p;

  if (tokenready)                         // is a token allready waiting?
    {
      tokenready = false;
      return true;
    }

  if (script->script_p >= script->end_p)
    return EndOfScript (crossline);

  //
  // skip space
  //
skipspace:
  while (*script->script_p <= 32)
    {
      if (script->script_p >= script->end_p)
	return EndOfScript (crossline);
      if (*script->script_p++ == '\n')
	{
	  if (!crossline)
	    fprintf (stderr, "Line %i is incomplete\n",scriptline);
	  scriptline = script->line++;
	}
    }

  if (script->script_p >= script->end_p)
    return EndOfScript (crossline);
  
  // ; # // comments
  if (*script->script_p == ';' || *script->script_p == '#'
      || ( script->script_p[0] == '/' && script->script_p[1] == '/') )
    {
      if (!crossline)
	fprintf (stderr, "Line %i is incomplete\n",scriptline);
      while (*script->script_p++ != '\n')
	if (script->script_p >= script->end_p)
	  return EndOfScript (crossline);
      goto skipspace;
    }
  
  // /* */ comments
  if (script->script_p[0] == '/' && script->script_p[1] == '*')
    {
      if (!crossline)
	fprintf (stderr, "Line %i is incomplete\n",scriptline);
      script->script_p+=2;
      while (script->script_p[0] != '*' && script->script_p[1] != '/')
	{
	  script->script_p++;
	  if (script->script_p >= script->end_p)
	    return EndOfScript (crossline);
	}
      script->script_p += 2;
      goto skipspace;
    }
  
  //
  // copy token
  //
  token_p = token;
  
  if (*script->script_p == '"')
    {
      // quoted token
      script->script_p++;
      while (*script->script_p != '"')
	{
	  *token_p++ = *script->script_p++;
	  if (script->script_p == script->end_p)
	    break;
	  if (token_p == &token[MAXTOKEN])
	    fprintf (stderr, "Token too large on line %i\n",scriptline);
	}
      script->script_p++;
    }
  else	// regular token
    while ( *script->script_p > 32 && *script->script_p != ';')
      {
	*token_p++ = *script->script_p++;
	if (script->script_p == script->end_p)
	  break;
	if (token_p == &token[MAXTOKEN])
	  fprintf (stderr, "Token too large on line %i\n",scriptline);
      }

  *token_p = 0;
  
  return true;
}


/*
==============
TokenAvailable

Returns true if there is another token on the line
==============
*/
int TokenAvailable (void)
{
  char    *search_p;

  search_p = script->script_p;

  if (search_p >= script->end_p)
    return false;

  while ( *search_p <= 32)
    {
      if (*search_p == '\n')
	return false;
      search_p++;
      if (search_p == script->end_p)
	return false;

    }

  if (*search_p == ';')
    return false;

  return true;
}


