/*
 ============================================================================
    $Revision: 1.0 $        $Date: 1997/11/20 15:01:26 $
 ============================================================================

                      PARADIGM SIMULATION, INCORPORATED
               Copyright (c) 1998 by Paradigm Simulation, Inc.

           This software is licensed and not sold. All use of this
           software is subject to the terms and conditions of the PSI
           software license, which should be read carefully. Copyright
           infringement is a serious and criminal offense.
           For more information, contact Paradigm Simulation at
           1-972-960-2301 or send e-mail to support@paradigmsim.com.

  ============================================================================
*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <time.h>
#include <assert.h>
//#include <Performer/pf.h>
#include <Performer/pr/pfLinMath.h>
#include <Performer/pr/pfGeoMath.h>
#include <Performer/pr/pfFlux.h>
#include <Performer/pr/pfEngine.h>
#include <Performer/pf/pfNode.h>
#include <Performer/pf/pfGeode.h>
#include <Performer/pf/pfGroup.h>
#include <Performer/pf/pfSwitch.h>
#include <Performer/pf/pfSequence.h>
#include <Performer/pr/pfGeoSet.h>
#include <Performer/pr/pfGeoState.h>
#include <Performer/pr/pfTexture.h>
#include <Performer/pf/pfTraverser.h>
#include <Performer/pfdu.h>

#include <ctype.h>

#include "pfmd2.h"
#include "animation.h"
#ifdef IRISGL
typedef signed char GLbyte;
typedef short GLshort;
typedef int GLint;
typedef unsigned char GLubyte;
typedef unsigned short GLushort;
typedef unsigned int GLuint;
typedef float GLfloat;
#else
#include <GL/gl.h>
#endif

static int gUserDataSlot = -1;
static int gPalettedTex = 0;


/*
 * Global loader modes.
 */
static int gAnimationType = PFMD2_ANIMATION_MORPH;
static int gDumpActions = PF_ON;
static float gQuakeUnitsToMeters = 1.0f;
//static float gQuakeUnitsToMeters = 0.04f; /* Convert Quake2 units to meters, approx. */
static int gZClamp = PF_ON;


#define IDALIASHEADER		(('2'<<24)+('P'<<16)+('D'<<8)+'I')
#define ALIAS_VERSION	8

#define	MAX_TRIANGLES	4096
#define MAX_VERTS		2048
#define MAX_FRAMES		512
#define MAX_MD2SKINS	32
#define	MAX_SKINNAME	64

typedef struct
{
  GLubyte  v[3];	    /* scaled byte to fit in frame mins/maxs */
  GLubyte  lightnormalindex;
} dtrivertx_t;

typedef struct
{
  GLfloat       scale[3];	/* multiply byte verts by this */
  GLfloat	translate[3];	/* then add this */
  char		name[16];	/* frame name from grabbing */
  dtrivertx_t	verts[1];	/* variable sized */
} daliasframe_t;

typedef struct
{
  GLshort s;
  GLshort t;
} dstvert_t;

typedef struct 
{
  GLshort	index_xyz[3];
  GLshort	index_st[3];
} dtriangle_t;

typedef struct
{
  GLint	ident;
  GLint	version;

  GLint	skinwidth;
  GLint skinheight;
  GLint framesize;		/* byte size of each frame */
  
  GLint num_skins;
  GLint num_xyz;
  GLint num_st;			/* greater than num_xyz for seams */
  GLint num_tris;
  GLint num_glcmds;		/* dwords in strip/fan command list */
  GLint num_frames;

  GLint ofs_skins;		/* each skin is a MAX_SKINNAME string */
  GLint ofs_st;			/* byte offset from start for stverts */
  GLint ofs_tris;		/* offset for dtriangles */
  GLint ofs_frames;		/* offset for first frame */
  GLint ofs_glcmds;	
  GLint ofs_end;		/* end of file */

} dmdl_t;

typedef struct
{
  pfFlux *frame_flux;
  pfFlux *vertex_flux;
  int num_iterations;
  pfEngine *vertex_engine;
} md2_user_data_t;

#define PRINT_PFVEC2(_level, _vec, _cmt) \
    (pfNotify((_level), PFNFY_PRINT, "%s: (%f, %f)", (_cmt), (_vec)[0], (_vec)[1]))

#define PRINT_PFVEC3(_level, _vec, _cmt) \
    (pfNotify((_level), PFNFY_PRINT, "%s: (%f, %f, %f)", (_cmt), (_vec)[0], (_vec)[1], (_vec)[2]))

#define PRINT_PFVEC4(_level, _vec, _cmt) \
    (pfNotify((_level), PFNFY_PRINT, "%s: (%f, %f, %f, %f)", (_cmt), (_vec)[0], (_vec)[1], (_vec)[2], (_vec)[3]))


static pfTexture *_loadPCX(char *name);

static void swap2(GLushort *array, GLuint length) {
// PAUL!
#ifndef Linux 
  GLushort tmp;
  GLubyte *ptr;
  
  while (length--) {
    tmp = *array;
    *array = ((tmp & 0xff) << 8) | (tmp >> 8);
    array++;
  }
#endif
}

static void swap4(GLuint *array, GLuint length)
{
// PAUL!!
#ifndef Linux
  GLuint tmp;
  while (length--) {
    tmp = *array;
    *array = (tmp << 24) | ((tmp << 8) & (0x00ff0000)) |
      ((tmp >> 8) & 0x0000ff00) | (tmp >> 24);    
    array++;
  }
#endif
}

//
// Animation loop engine source numbers.
//
#define PFENG_ANIMLOOP_FRAME          0
#define PFENG_ANIMLOOP_LOOPBOUNDS     1
#define PFENG_ANIMLOOP_SRC_0          2
#define PFENG_ANIMLOOP_SRC(n)         (PFENG_ANIMLOOP_SRC_0 + n)


static void anim_loop_engine(pfEngine *eng)
{
  int i, j;
  void *dst, *src0, *src1, *src2, *fsrc0, *fsrc1;
  float *d, *frame, *loop, *bounds, *f0, *f1;
  int d_stride, s_stride0, s_stride1, s_stride2, f_stride0, f_stride1;
  int d_offset, s_offset0, s_offset1, s_offset2, f_offset0, f_offset1;
  ushort *d_ilist, *s_ilist0, *s_ilist1, *s_ilist2, *f_ilist0, *f_ilist1;
  int s_icount0, s_icount1, s_icount2, f_icount0, f_icount1;
  int iterations, items;

  eng->getDst(&dst, &d_ilist, &d_offset, &d_stride);

  int num_srcs = eng->getNumSrcs();

  if (num_srcs < 3 || dst == NULL ){
    pfNotify(PFNFY_WARN, PFNFY_PRINT, 
	     "libpfmd2: Error animation loop engine not configured properly.");
    return;
  }

  eng->getSrc(PFENG_ANIMLOOP_FRAME, &src0, &s_ilist0, &s_icount0, &s_offset0,
	      &s_stride0);
  eng->getSrc(PFENG_ANIMLOOP_LOOPBOUNDS, &src1, &s_ilist1, &s_icount1, 
	      &s_offset1, &s_stride1);

  eng->getIterations(&iterations, &items);

  d = (float*)((pfFlux*)dst)->getWritableData();
  if (d == NULL)
    return;
  d += d_offset;

  if (pfMemory::isOfType(src0, pfFlux::getClassType()))
    frame = (float *)((pfFlux *)src0)->getCurData();
  else
    frame = (float *)src0;
  frame += s_offset0;

  if (pfMemory::isOfType(src1, pfFlux::getClassType()))
    bounds = (float *)((pfFlux *)src1)->getCurData();
  else
    bounds = (float *)src1;
  bounds += s_offset1;

  // Get the src's to lerp between
  float weight = frame[0] - floorf(frame[0]);

  // Determine frame0 and frame1.  Handle wrapping.
  int f0ndx = floorf(frame[0]);
  int f1ndx = f0ndx+1;
  int start = bounds[0];
  int end = bounds[1];

  if (f1ndx >= end) f1ndx = start;

  eng->getSrc(PFENG_ANIMLOOP_SRC(f0ndx), &fsrc0, &f_ilist0, &f_icount0, 
	      &f_offset0, &f_stride0);
  eng->getSrc(PFENG_ANIMLOOP_SRC(f1ndx), &fsrc1, &f_ilist1, &f_icount1, 
	      &f_offset1, &f_stride1);

  if (pfMemory::isOfType(fsrc0, pfFlux::getClassType()))
    f0 = (float *)((pfFlux *)fsrc0)->getCurData();
  else
    f0 = (float *)fsrc0;
  f0 += f_offset0;

  if (pfMemory::isOfType(fsrc1, pfFlux::getClassType()))
    f1 = (float *)((pfFlux *)fsrc1)->getCurData();
  else
    f1 = (float *)fsrc1;
  f1 += f_offset1;

  for (i = 0; i < iterations; i++) {
    for (j = 0; j < items; j++) {
      d[j] = f0[j] * (1.0f - weight) + f1[j] * weight;
    }
    d += d_stride;
    f0 += f_stride0;
    f1 += f_stride1;
  }

}


static int _getFrameNum(char name[16])
{
  int num;
  char *fram_num;
  
  fram_num = strpbrk(name, "0123456789");
  num = atoi(fram_num);
  return num;
}

static void _getActionName(char name[16], char action[16])
{
  char *fram_num;
  strcpy(action, name);
  fram_num = strpbrk(action, "0123456789");
  if (fram_num)
    fram_num[0] = '\0';
}

typedef struct {
  pfVec2 st;
  pfVec3 xyz;
} md2Vertex;

static float _computeZOffset(daliasframe_t *frame, int framesize, 
			     int num_frames, int num_verts)
{
  int i, j;
  float min_z = 1000000.0f;
  float max_z = -min_z;
  dtrivertx_t *vrt;
  float z;

  /* Use the first frame named 'stand' to determine z-offset. */
  for (i = 0; i < num_frames; i++) {
    daliasframe_t *cur_frame = (daliasframe_t *)((GLubyte *)frame + framesize * i);
    if (strstr(cur_frame->name, "stand")) {
      for (j = 0; j < num_verts; j++) {
	vrt = &(cur_frame->verts[j]);
	z = ((float)(vrt->v[2]) * cur_frame->scale[2] + cur_frame->translate[2]);
	if (z < min_z) min_z = z;
	if (z > max_z) max_z = z;
      }
      return min_z;
    }
  }

  /* If no frame 'stand' found use the minimum z value. */
  for (i = 0; i < num_frames; i++) {
    daliasframe_t *cur_frame = (daliasframe_t *)((GLubyte *)frame + framesize * i);
    for (j = 0; j < num_verts; j++) {
      vrt = &(cur_frame->verts[j]);
      z = ((float)(vrt->v[2]) * cur_frame->scale[2] + cur_frame->translate[2]);
      if (z < min_z) min_z = z;
      if (z > max_z) max_z = z;
    }
  }

  return min_z;
}

static void _applyZOffset(float z_offset, daliasframe_t *frame, 
			  int framesize, int num_frames)
{
  int i, j;
  dtrivertx_t *vrt;
  float z;

  for (i = 0; i < num_frames; i++) {
    daliasframe_t *cur_frame = (daliasframe_t *)((GLubyte *)frame + framesize * i);
    cur_frame->translate[2] -= z_offset;
  }

}

static int _countFramesInAction(daliasframe_t *frame, int remaining_frames, int framesize) 
{
  int total_frames = 1;
  daliasframe_t *cur_frame = frame;
  int cur_frame_num = -1;
  int last_frame_num = MAX_FRAMES+1;
  int i;

  last_frame_num = _getFrameNum(cur_frame->name);
  cur_frame = (daliasframe_t *)((GLubyte *)cur_frame + framesize);
  for (i = 1; i < remaining_frames; i++) {
    cur_frame_num = _getFrameNum(cur_frame->name);
    if (cur_frame_num != last_frame_num+1) {
      return total_frames;
    }
    last_frame_num = cur_frame_num;
    total_frames++;
    cur_frame = (daliasframe_t *)((GLubyte *)cur_frame + framesize);
  }
  return total_frames;
}


static void _addVertexToEng(pfEngine *eng, daliasframe_t *frame, 
			    int num_verts, int frame_num, pfFlux *vflux, pfGeode *geode)
{
  pfVec3 *v, xyz;
  int i;
  dtrivertx_t *vrt;
  pfSphere bsph;

  v = (pfVec3 *)pfMalloc(sizeof(pfVec3) * num_verts, pfGetSharedArena());
  for (i = 0; i < num_verts; i++) {
    vrt = &(frame->verts[i]);
    xyz[0] = ((float)(vrt->v[0]) * frame->scale[0] + frame->translate[0]);
    xyz[1] = ((float)(vrt->v[1]) * frame->scale[1] + frame->translate[1]);
    xyz[2] = ((float)(vrt->v[2]) * frame->scale[2] + frame->translate[2]);
    v[i].scale(gQuakeUnitsToMeters, xyz);
  }

  if (frame_num == 0) {
    vflux->initData(v);
  }

  if (eng) {
    eng->setSrc( PFENG_MORPH_SRC(frame_num), v, NULL, 0, 0, 3);
  }

/*** PAUL!!! - have isect testing on primitive level
  if (frame_num == 0) {
    bsph.around((const pfVec3 *)v, num_verts);
    geode->setBound(&bsph, PFBOUND_STATIC);
  }
****/
}


static pfNode *_convertFrame(daliasframe_t *frame, GLint *glcmds, 
			     pfGeoState *gst, int num_verts)
{
  GLint *command = glcmds;
  int num_prim_verts;
  pfGeode *geode;
  void *arena = pfGetSharedArena();
  pfVec3 *v, xyz, pnt;
  pfVec2 *t, st;
  pfVec4 *c;
  ushort *vi, *ti, *ci;
  pfGeoSet *gset;
  int *lens;
  int i, ndx;
  dtrivertx_t *vrt;
  int st_count;
  pfSphere bsph;
  pfBox bbox;

  c = (pfVec4 *)pfMalloc(sizeof(pfVec4), arena);
  ci = (ushort *)pfMalloc(sizeof(ushort), arena);
  c[0].set( 1.0f, 1.0f, 1.0f, 1.0f);
  ci[0] = 0;

  v = (pfVec3 *)pfMalloc(sizeof(pfVec3) * num_verts, arena);
  for (i = 0; i < num_verts; i++) {
    vrt = &(frame->verts[i]);
    xyz[0] = ((float)(vrt->v[0]) * frame->scale[0] + frame->translate[0]);
    xyz[1] = ((float)(vrt->v[1]) * frame->scale[1] + frame->translate[1]);
    xyz[2] = ((float)(vrt->v[2]) * frame->scale[2] + frame->translate[2]);
    v[i].scale( gQuakeUnitsToMeters, xyz);
  }

  geode = new pfGeode();
  bsph.around( (const pfVec3 *)v, num_verts);
  //  geode->setBound(&bsph, PFBOUND_STATIC);

  command = glcmds;
  st_count = 0;

  while (*command) {
    if (*command > 0) {

      /* triangle strip */
      num_prim_verts = *command; command++;

      gset = new pfGeoSet();
      gset->setGState(gst);

      lens = (int *)pfMalloc(sizeof(int), arena);
      lens[0] = num_prim_verts;

      vi = (ushort *)pfMalloc(sizeof(ushort) * num_prim_verts, arena);
      t = (pfVec2 *)pfMalloc(sizeof(pfVec2) * num_prim_verts, arena);
      ti = (ushort *)pfMalloc(sizeof(ushort) * num_prim_verts, arena);
 
      gset->setPrimType(PFGS_TRISTRIPS);
      gset->setNumPrims(1);
      gset->setPrimLengths(lens);

      for (i = 0; i < num_prim_verts; i++) {
	st[0] = *(float *)(command); command++;
	st[1] = *(float *)(command); command++;
	t[i].copy(st);
	ti[i] = i;
	vi[i] = *command; command++;
	pnt.copy(v[vi[i]]);
	bbox.extendBy(pnt);
      }

      gset->setAttr(PFGS_COLOR4, PFGS_OVERALL, c, ci);
      gset->setAttr(PFGS_COORD3, PFGS_PER_VERTEX, v, vi);
      gset->setAttr(PFGS_TEXCOORD2, PFGS_PER_VERTEX, t, ti);

      geode->addGSet(gset);

    } else {
      /* triangle fan */
      num_prim_verts = *command; command++;
      num_prim_verts = -num_prim_verts;

      gset = new pfGeoSet();
      gset->setGState(gst);

      lens = (int *)pfMalloc(sizeof(int), arena);
      lens[0] = num_prim_verts;

      vi = (ushort *)pfMalloc(sizeof(ushort) * num_prim_verts, arena);
      t = (pfVec2 *)pfMalloc(sizeof(pfVec2) * num_prim_verts, arena);
      ti = (ushort *)pfMalloc(sizeof(ushort) * num_prim_verts, arena);

      gset->setPrimType(PFGS_TRIFANS);
      gset->setNumPrims(1);
      gset->setPrimLengths(lens);

      for (i = 0; i < num_prim_verts; i++) {
	st[0] = *(float *)(command); command++;
	st[1] = *(float *)(command); command++;
	t[i].copy(st);
	ti[i] = i;
	vi[i] = *command; command++;
	pnt.copy(v[vi[i]]);
	bbox.extendBy(pnt);
      }

      gset->setAttr(PFGS_COLOR4, PFGS_OVERALL, c, ci);
      gset->setAttr(PFGS_COORD3, PFGS_PER_VERTEX, v, vi);
      gset->setAttr(PFGS_TEXCOORD2, PFGS_PER_VERTEX, t, ti);

      geode->addGSet(gset);

    }
  }

  return (pfNode *)geode;
}


static pfGeode *_createFluxedGeode(int num_verts, daliasframe_t *frame, 
				   pfGeoState *gst, GLint *glcmds, pfFlux *vertex_flux)
{
  GLint *command = glcmds;
  int num_prim_verts;
  pfGeode *geode;
  void *arena = pfGetSharedArena();
  pfVec3 *v, xyz, pnt;
  pfVec2 *t, st;
  pfVec4 *c;
  ushort *vi, *ti, *ci;
  pfGeoSet *gset;
  int *lens;
  int i, ndx;
  dtrivertx_t *vrt;
  int st_count;
  pfSphere bsph;
  pfBox bbox;

  geode = new pfGeode();

  c = (pfVec4 *)pfMalloc(sizeof(pfVec4), arena);
  ci = (ushort *)pfMalloc(sizeof(ushort), arena);
  c[0].set( 1.0f, 1.0f, 1.0f, 1.0f);
  ci[0] = 0;

  v = (pfVec3 *)pfMalloc(sizeof(pfVec3) * num_verts, pfGetSharedArena());
  for (i = 0; i < num_verts; i++) {
    vrt = &(frame->verts[i]);
    xyz[0] = ((float)(vrt->v[0]) * frame->scale[0] + frame->translate[0]);
    xyz[1] = ((float)(vrt->v[1]) * frame->scale[1] + frame->translate[1]);
    xyz[2] = ((float)(vrt->v[2]) * frame->scale[2] + frame->translate[2]);
    v[i].scale(gQuakeUnitsToMeters, xyz);
  }

  command = glcmds;
  st_count = 0;

  while (*command) {
    if (*command > 0) {

      /* triangle strip */
      num_prim_verts = *command; command++;

      gset = new pfGeoSet();
      gset->setGState(gst);

      lens = (int *)pfMalloc(sizeof(int), arena);
      lens[0] = num_prim_verts;

      vi = (ushort *)pfMalloc(sizeof(ushort) * num_prim_verts, arena);
      t = (pfVec2 *)pfMalloc(sizeof(pfVec2) * num_prim_verts, arena);
      ti = (ushort *)pfMalloc(sizeof(ushort) * num_prim_verts, arena);

      gset->setPrimType(PFGS_TRISTRIPS);
      gset->setNumPrims(1);
      gset->setPrimLengths(lens);

      for (i = 0; i < num_prim_verts; i++) {
	st[0] = *(float *)(command); command++;
	st[1] = *(float *)(command); command++;
	t[i].copy(st);
	ti[i] = i;
	vi[i] = *command; command++;
	pnt.copy(v[vi[i]]);
	bbox.extendBy(pnt);
      }

      gset->setAttr(PFGS_COLOR4, PFGS_OVERALL, c, ci);
      gset->setAttr(PFGS_COORD3, PFGS_PER_VERTEX, vertex_flux, vi);
      gset->setAttr(PFGS_TEXCOORD2, PFGS_PER_VERTEX, t, ti);

      //      gset->setBound(&bbox, PFBOUND_STATIC);
      geode->addGSet(gset);

    } else {
      /* triangle fan */
      num_prim_verts = *command; command++;
      num_prim_verts = -num_prim_verts;

      gset = new pfGeoSet();
      gset->setGState(gst);

      lens = (int *)pfMalloc(sizeof(int), arena);
      lens[0] = num_prim_verts;

      vi = (ushort *)pfMalloc(sizeof(ushort) * num_prim_verts, arena);
      t = (pfVec2 *)pfMalloc(sizeof(pfVec2) * num_prim_verts, arena);
      ti = (ushort *)pfMalloc(sizeof(ushort) * num_prim_verts, arena);

      gset->setPrimType(PFGS_TRIFANS);
      gset->setNumPrims(1);
      gset->setPrimLengths(lens);

      bbox.makeEmpty();
      
      for (i = 0; i < num_prim_verts; i++) {
	st[0] = *(float *)(command); command++;
	st[1] = *(float *)(command); command++;
	t[i].copy(st);
	ti[i] = i;
	vi[i] = *command; command++;
	pnt.copy(v[vi[i]]);
	bbox.extendBy(pnt);
      }

      gset->setAttr( PFGS_COLOR4, PFGS_OVERALL, c, ci);
      gset->setAttr( PFGS_COORD3, PFGS_PER_VERTEX, vertex_flux, vi);
      gset->setAttr( PFGS_TEXCOORD2, PFGS_PER_VERTEX, t, ti);

      //      gset->setBound( &bbox, PFBOUND_STATIC);

      geode->addGSet(gset);

    }
  }

  pfDelete(v);

  return geode;
}

static void _disableFlux(pfNode *node)
{
  md2_user_data_t *ud = (md2_user_data_t *)node->getUserData(gUserDataSlot);
  if (ud && ud->vertex_engine) {
    ud->vertex_engine->setIterations(0,3);
  }

}

static void _enableFlux(pfNode *node)
{
  md2_user_data_t *ud = (md2_user_data_t *)node->getUserData(gUserDataSlot);
  if (ud && ud->vertex_engine) {
    ud->vertex_engine->setIterations(ud->num_iterations, 3);    
  }

}
static int _actionControl(pfTraverser *trav, void *data)
{
  pfNode *me = trav->getNode();
  md2Animation *anim = (md2Animation *)me->getUserData(gUserDataSlot);
  
  // Update frame control flux to play current sequence.
  anim->update();

  return PFTRAV_CONT;
}

static int _switchFluxControl(pfTraverser *trav, void *data)
{
  pfNode *child;
  pfNode *me = trav->getNode();
  int i;

  if (me->isOfType(pfSwitch::getClassType())) {
    int val = ((pfSwitch *)me)->getVal();
    int num_kids = ((pfSwitch *)me)->getNumChildren();
    for (i = 0; i < num_kids; i++) {
      /* Turn off children fluxes. */
      child = ((pfSwitch *)me)->getChild(i);
      _disableFlux(child);
    }
    
    /* Enable active child's flux. */
    child = ((pfSwitch *)me)->getChild(val);
    _enableFlux(child);

  } else {
    pfNotify(PFNFY_WARN, PFNFY_PRINT, "pfdLoadFile_md2: Unexpected node.");
  }
  return PFTRAV_CONT;
}

extern void
pfdInitConverter_md2(void)
{
  gUserDataSlot = pfObject::getNamedUserDataSlot("MD2 Loader");
}

extern void    
pfdConverterMode_md2(int mode, int value)
{
  switch (mode) {
  case PFMD2_ANIMATION:
    gAnimationType = value;
    break;
  case PFMD2_DUMP_ACTIONS:
    gDumpActions = value;
    break;
  case PFMD2_Z_CLAMP:
    gZClamp = value;
    break;

  default:
    pfNotify(PFNFY_WARN, PFNFY_USAGE,
	     "pfdConverterMode_md2: Unknown mode %d", mode);
  }
}

extern int     
pfdGetConverterMode_md2(int mode)
{
  switch (mode) {
  case PFMD2_ANIMATION:
    return gAnimationType;
  case PFMD2_DUMP_ACTIONS:
    return gDumpActions;
  case PFMD2_Z_CLAMP:
    return gZClamp;
  case PFMD2_USER_DATA_SLOT:
    return gUserDataSlot;

  default:
    pfNotify(PFNFY_WARN, PFNFY_USAGE,
	     "pfdConverterMode_md2: Unknown mode %d", mode);
    return -1;
  }
}

extern void    
pfdConverterVal_md2(int which, float val)
{
  switch (which) {
  default:
    pfNotify(PFNFY_WARN, PFNFY_USAGE,
	     "pfdConverterVal_md2: Unknown value %d", which);
  }
}


extern float   
pfdGetConverterVal_md2(int which)
{
  switch (which) {
  default:
    pfNotify(PFNFY_WARN, PFNFY_USAGE,
	     "pfdConverterVal_md2: Unknown value %d", which);
    return (-1.0f);
  }
}

extern void    
pfdConverterAttr_md2(int which, void *attr)
{
  switch (which) {
  default:
    pfNotify(PFNFY_WARN, PFNFY_USAGE,
	     "pfdConverterAttr_md2: Unknown attribute %d", which);
  }
}

extern void *   
pfdGetConverterAttr_md2(int which)
{
  switch (which) {
  default:
    pfNotify(PFNFY_WARN, PFNFY_USAGE,
	     "pfdConverterAttr_md2: Unknown attribute %d", which);
    return NULL;
  }
}



extern pfNode *
pfdLoadFile_md2(const char *fileName)
{
  FILE	        *fp	  = NULL;
  FILE          *ofp      = NULL;
  pfGroup       *root	  = NULL;
  GLint          num_glcmds;
  GLint         *glcmds;
  dmdl_t         hdr;
  pfNode        *node	  = NULL;
  GLint          framesize;
  GLint          num_frames;
  daliasframe_t *frames;
  GLint          num_skins;
  char         **skin_names;
  pfNode        *frm_node;
  pfSequence    *frm_seq;
  pfGeoState    *gst;
  int            i, len;
  pfTexture     *tex;
  pfTexEnv      *tex_env;
  char          *tex_name, *last_slash, *last_dot, *tmpSkin;
  char           action_name[16];
  int            frame_num, last_frame_num;
  pfSwitch      *frm_switch;
  int            new_action;
  pfEngine      *vertex_engine, *time_engine;
  pfFlux        *frame_flux, *vertex_flux, *time_frame_flux, *time_scale_flux;
  int            frame_count = 0;
  pfGeode       *frm_geom;
  md2_user_data_t *frm_ud;
  float          z_offset;
  pfFlux        *loop_flux, *loop_bounds_flux;

  if ((fp = pfdOpenFile(fileName)) == NULL)
    return NULL;

  fread(&hdr, sizeof(GLint), 17, fp);

  /* Fix up endedness. */
  swap4((GLuint *)&hdr, 17);

  /* Read GL commands. */
  num_glcmds = hdr.num_glcmds;
  fseek(fp, hdr.ofs_glcmds, SEEK_SET);
  glcmds = (GLint *)pfMalloc(num_glcmds * sizeof(GLint),
			      pfGetSharedArena());
  fread(glcmds, num_glcmds * sizeof(GLint), 1, fp);
  swap4((GLuint *)glcmds, num_glcmds);

  /* Read frame data. */
  framesize = hdr.framesize;
  num_frames = hdr.num_frames;
  fseek(fp, hdr.ofs_frames, SEEK_SET);
  frames = (daliasframe_t *)pfMalloc(hdr.framesize * hdr.num_frames,
				      pfGetSharedArena());
  fread(frames, framesize * num_frames, 1, fp);  

  for (i = 0; i < num_frames; i++) {
    daliasframe_t *frame = (daliasframe_t *)((GLubyte *)frames + framesize * i);
    swap4((GLuint *)frame->scale, 6);
  }

  /* Read skin (texture) names. */
  fseek(fp, hdr.ofs_skins, SEEK_SET);
  num_skins = hdr.num_skins;
  
  if (num_skins) {
    skin_names = (char **)pfMalloc(sizeof(char *) * num_skins,
				    pfGetSharedArena());
    for (i=0; i < num_skins; i++) {
      skin_names[i] = (char *)pfMalloc(MAX_SKINNAME, pfGetSharedArena());
      fread(skin_names[i], MAX_SKINNAME, 1, fp);
    }
  } else {
    skin_names = NULL;
  }

  root = new pfGroup();

  gst = new pfGeoState();
  gst->setMode(PFSTATE_ENLIGHTING, PF_OFF);
  gst->setMode(PFSTATE_CULLFACE, PFCF_FRONT);

  /* Build texture name. */
  len = strlen(fileName) + 1;
  tex_name = (char *)pfMalloc(sizeof(char) * len, pfGetSharedArena());

  /* Load the first texture by default */
  // PAUL - if skin name is NULL, load skin.pcx in same dir
  if (skin_names == NULL || skin_names[0][0] == '\0') {
    tmpSkin = strdup(fileName);
    last_slash = strrchr(tmpSkin, '/'); 
    if (*last_slash == '\0') {
      //fprintf(stderr, "no skin... using skin.pcx\n");
      tex = _loadPCX("skin.pcx");
    }
    else {
      *last_slash = '\0';
      strcat(tmpSkin, "/skin.pcx");
      //fprintf(stderr, "no skin... using %s\n", tmpSkin);
      tex = _loadPCX(tmpSkin);
    }
  }
  else 
    tex = _loadPCX(skin_names[0]);

  gst->setAttr(PFSTATE_TEXTURE, (void *)tex);
  gst->setMode(PFSTATE_ENTEXTURE, PF_ON);

  tex_env = new pfTexEnv();
  tex_env->setMode( PFTE_REPLACE);
  gst->setAttr(PFSTATE_TEXENV, (void *)tex_env);

  /* Convert frame. */
  last_slash = strrchr(fileName, '/');
  if (last_slash) strcpy(tex_name, last_slash);
  else strcpy(tex_name, fileName);
  last_dot = strrchr(tex_name, '.');
  last_dot[0] = '\0';

  if (gAnimationType == PFMD2_ANIMATION_SEQUENCE) {
    frm_switch = new pfSwitch();
    frm_switch->setName(tex_name);
    root->addChild(frm_switch);
  }

  action_name[0] = '\0';
  last_frame_num = 10000;

  /* Create info file if needed. */
  if (gDumpActions == PF_ON) {
    strcat(tex_name, ".info");
    ofp = fopen(tex_name, "wa");
  }

  if (ofp) {
    fprintf(ofp, "Quake2 Model: %s  Total Frames: %d\n", 
	    fileName, num_frames);
  }

  if (gZClamp == PF_ON) {
    /* Compute z-offset to force minimum z value to be 0.0 */
    z_offset = _computeZOffset(frames, framesize, 
			       num_frames, hdr.num_xyz);
    _applyZOffset(z_offset, frames, framesize, num_frames);
  }

  if (gAnimationType == PFMD2_ANIMATION_MORPH) {

    /* Create frame counter flux to drive animation. */
    frame_flux = new pfFlux(sizeof(float), PFFLUX_DEFAULT_NUM_BUFFERS);
    frame_flux->setMode(PFFLUX_PUSH, PF_ON);
    
    /* Create vertex flux to hold output of vertex engine. */
    vertex_flux = new pfFlux(sizeof(pfVec3) * hdr.num_xyz, PFFLUX_DEFAULT_NUM_BUFFERS);

    if (num_frames > 1) {

      float *time_scale = NULL;

      /* Get the time flux and set up scale factors. */
      time_frame_flux = pfGetFrameTimeFlux();
      time_scale_flux = new pfFlux(sizeof(float)*4, PFFLUX_DEFAULT_NUM_BUFFERS);
      time_scale = (float *)pfMalloc(sizeof(float)*4, pfGetSharedArena());
      time_scale[0] = 0.0f;
      time_scale[1] = 6.0f;
      time_scale[2] = (float)num_frames - 1;
      time_scale[3] = 0.0f;
      time_scale_flux->initData(time_scale);

      time_engine = new pfEngine(PFENG_TIME);
      time_engine->setSrc(PFENG_TIME_TIME, time_frame_flux, NULL, 0, 0, 0);
      time_engine->setDst(frame_flux, NULL, 0, 0);
      time_engine->setSrc(PFENG_TIME_SCALE, time_scale_flux, NULL, 0, 0, 0);
      time_engine->setMode(PFENG_TIME_MODE, PFENG_TIME_CYCLE);

#ifdef USE_MORTH_ENG
      vertex_engine = new pfEngine(PFENG_MORPH);
      vertex_engine->setSrc(PFENG_MORPH_FRAME, frame_flux, NULL, 0, 0, 0);
      vertex_engine->setDst(vertex_flux, NULL, 0, 3);
      vertex_engine->setIterations(hdr.num_xyz, 3);
#else

      loop_bounds_flux = new pfFlux(sizeof(float) * 2, 
				    PFFLUX_DEFAULT_NUM_BUFFERS);
      vertex_engine = new pfEngine(PFENG_USER_FUNCTION);
      vertex_engine->setUserFunction(anim_loop_engine);
      vertex_engine->setSrc(PFENG_ANIMLOOP_FRAME, frame_flux, NULL, 0, 0, 0);
      vertex_engine->setSrc(PFENG_ANIMLOOP_LOOPBOUNDS, loop_bounds_flux, NULL,
			    0, 0, 0);
      vertex_engine->setDst(vertex_flux, NULL, 0, 3);
      vertex_engine->setIterations(hdr.num_xyz, 3);
#endif
    } else {
      vertex_engine = NULL;
    }

    /* Create a pfGeode with pfFluxed geosets for this entire action. */
    frm_geom = _createFluxedGeode(hdr.num_xyz, frames, gst,
				  glcmds, vertex_flux);

    /* Add the engine and flux info to the geod as user data. */
    frm_ud = (md2_user_data_t *)pfMalloc(sizeof(md2_user_data_t),
					 pfGetSharedArena());
    frm_ud->frame_flux = frame_flux;
    frm_ud->vertex_flux = vertex_flux;
    frm_ud->vertex_engine = vertex_engine;
    frm_ud->num_iterations = hdr.num_xyz;
    frm_geom->setUserData(gUserDataSlot, (void *)frm_ud);

    frm_geom->setName(action_name);
    root->addChild(frm_geom);
  }

  for (i = 0; i < num_frames; i++) {
    daliasframe_t *frame = (daliasframe_t *)((GLubyte *)frames + framesize * i);

    if (gAnimationType == PFMD2_ANIMATION_MORPH) {

      _getActionName(frame->name, action_name);
      if (ofp) {
	fprintf(ofp, "Action: %s  Frames: %d\n", 
		action_name, i);
      }

      /* Add current vertex frame to engine. */
      _addVertexToEng(vertex_engine, frame, hdr.num_xyz, i, vertex_flux, frm_geom);

    } else {

      frame_num = _getFrameNum(frame->name);

      if (frame_num != (last_frame_num+1)) {
	int total_frames = _countFramesInAction(frame, num_frames - i, 
						framesize);
	frm_seq = new pfSequence();
	_getActionName(frame->name, action_name);
	if (ofp) {
	  fprintf(ofp, "Action: %s  Num Frames: %d\n", 
		  action_name, total_frames);
	}
	frm_seq->setName(action_name);
	frm_switch->addChild(frm_seq);
	frm_seq->setTime(-1, 0.15);
	frm_seq->setInterval(PFSEQ_CYCLE, 0, -1);
	frm_seq->setDuration(1.0f, -1.0f);
	frm_seq->setMode(PFSEQ_START);
      }
      last_frame_num = frame_num;
      frm_node = _convertFrame(frame, glcmds, gst, hdr.num_xyz);

      frm_seq->addChild(frm_node);
    }

  }

  if (gAnimationType == PFMD2_ANIMATION_SEQUENCE) {
    frm_switch->setVal(0);
    /* Add APP callback to switch to disable non-visible fluxes. */
    frm_switch->setTravFuncs( PFTRAV_APP, _switchFluxControl, NULL);
  } else if (gAnimationType == PFMD2_ANIMATION_MORPH) {

    float *bounds = (float *)pfMalloc(sizeof(float)*2, pfGetSharedArena());
    if (num_frames > 1) {
      bounds[0] = 0.0f;
      bounds[1] = (float)(num_frames-1);
      loop_bounds_flux->initData(bounds);
    }

    md2Animation *anim = new md2Animation();
    anim->setNumFrames(num_frames);
    anim->setFrameControl(time_scale_flux, loop_bounds_flux);
    anim->addAction(0, num_frames-1);
    root->setUserData(gUserDataSlot, (void *)anim);
    root->setTravFuncs(PFTRAV_APP, _actionControl, NULL);
  }

  if (ofp) fclose(ofp);
  
  return (pfNode *)root;
    
} /* pfdLoadFile_md2 */


extern int
pfdStoreFile_md2(pfNode *root, const char *fileName)
{
  return 0;
} /* pfdStoreFile_md2 */

extern md2Animation *
md2GetNodeAnimation(pfNode *md2node)
{
  md2Animation *anim = (md2Animation *)md2node->getUserData(gUserDataSlot);
  return anim;
}

typedef struct
{
  char	manufacturer;
  char	version;
  char	encoding;
  char	bits_per_pixel;
  unsigned short	xmin,ymin,xmax,ymax;
  unsigned short	hres,vres;
  unsigned char	palette[48];
  char	reserved;
  char	color_planes;
  unsigned short	bytes_per_line;
  unsigned short	palette_type;
  char	filler[58];
  unsigned char	data;			// unbounded
} pcx_t;


static pfTexture *_loadPCX(char *name)
{
  FILE *fp;
  uint *image;
  char *fullName = NULL;
  pcx_t hdr;
  int i;
  static pfList *texList = NULL;  

  if (texList == NULL) 
    texList = new pfList(sizeof(pfTexture *), 3);

  // Convert to all lower-case letters.
  //  _toLower(name);
  int len = strlen(name);
  fullName = (char *)pfMalloc(sizeof(char)*(len+1), pfGetSharedArena());
  for (i = 0; i < len; i++) {
    name[i] = tolower(name[i]);
    fullName[i] = name[i];
    if (fullName[i] == '/') fullName[i] = '_';
  }
  // PAUL!!!
  fullName[i] = '\0';

  if (name[0] == '\0')
    name = strdup("skin.pcx");

  // print out tex 
  //fprintf(stderr, "loading tex: %s\n", name);

  // Determine if the texture has already been loaded.
  int numTex = texList->getNum();
  for (i = 0; i < numTex; i++) {
    pfTexture *oldtex = (pfTexture *)texList->get(i);
    if (!strcmp(oldtex->getName(), fullName)) {
      return oldtex;
    }
  }

  if ((fp = pfdOpenFile(name)) == NULL) {
     if ((fp = pfdOpenFile("skin.pcx")) == NULL)
       return NULL;
  }

  //
  // Read the PCX image file.
  //
  fread(&hdr, sizeof(pcx_t), 1, fp);

  swap2((GLushort *)&hdr.xmin, 6);
  swap2((GLushort *)&hdr.bytes_per_line, 2);

  if (hdr.manufacturer != 0x0a
      || hdr.version != 5
      || hdr.encoding != 1
      || hdr.bits_per_pixel != 8
      || hdr.xmax >= 640
      || hdr.ymax >= 480) {
    fprintf(stderr, "Unsupported PCX format. BPP: %d (8) Ver: %d (5) Enc: %d (1)\n",
	   hdr.bits_per_pixel, hdr.version, hdr.encoding);
    fclose(fp);
    return NULL;
  }

  int width = hdr.xmax+1;
  int height = hdr.ymax+1;

  GLubyte *pbuf = (GLubyte *)pfMalloc(sizeof(GLubyte) * width * height,
				      pfGetSharedArena());
  GLubyte dataByte;

  GLubyte *pix = pbuf;
  int runLength;
  int x, y;
  int totalBytes = 0;

  x = 0;
  while (x < width * height) {
    
    dataByte = fgetc(fp);
    if((dataByte & 0xC0) == 0xC0) {
      runLength = dataByte & 0x3F;
      dataByte = fgetc(fp);
    } else
      runLength = 1;

    while(runLength-- > 0)
      pix[x++] = dataByte;

  }

  // Read the palette.
  const int pal_len = 768;
  GLubyte *palette = (GLubyte *)pfMalloc(sizeof(GLubyte)*pal_len);

  fseek(fp,-pal_len,SEEK_END);
  fread(palette, 1, pal_len, fp);

  fclose(fp);

  // Create a pfTexture from the image.
  // Optionally support a paletted texture.

  pfTexture *tex = new pfTexture();

  // Don't use the real file name as the texture name.  Vega strips leading
  // path info off and optimizes away textures with the same name.
  // The fullName has _ substituted for /.
  tex->setName(fullName);

  if (gPalettedTex) {

  } else {

    GLuint *rgbaimage = (uint *)pfMalloc(sizeof(GLuint)*width*height, 
				 pfGetSharedArena());
    GLubyte *img = (GLubyte *)pbuf;
    GLubyte r, g, b, a;
    a = 255;
    for (i = 0; i < width * height; i++) {
      r = palette[img[i]*3+0];
      g = palette[img[i]*3+1];
      b = palette[img[i]*3+2];
#ifdef Linux
      rgbaimage[i] = a << 24 | b << 16 | g << 8 | r;
#else
      rgbaimage[i] = r << 24 | g << 16 | b << 8 | a;
#endif
    }

    tex->setImage(rgbaimage, 4, width, height, 0);
    tex->setFormat(PFTEX_INTERNAL_FORMAT, PFTEX_RGB5_A1);
    tex->setFormat(PFTEX_EXTERNAL_FORMAT, PFTEX_PACK_8);
    tex->setFilter(PFTEX_MINFILTER, PFTEX_MIPMAP_BILINEAR);
    tex->setFilter(PFTEX_MAGFILTER, PFTEX_BILINEAR);

  }

  texList->add((void *)tex);

  return tex;

}

