//====================================================================
//#
//#  CaveQuakeBullet.h                                ##   ##     
//#                                                   ##   ##
//#  Author(s): Paul Rajlich                          #######
//#  Email: prajlich@ncsa.uiuc.ed                     ##   ##
//#  Date created:                                    ##   ##
//#
//#  Description: CAVE QUAKE II
//#
//====================================================================
//#  (c)opyright 1998, 1999  Paul Rajlich, all rights reserved
//====================================================================

#ifndef CAVE_QUAKE_BULLET_H
#define CAVE_QUAKE_BULLET_H

#include "CaveQuakeOpponent.h"

extern pfGroup *levelGroup;
extern pfDCS *transDcs;
extern CaveQuakePlayer *player;
extern pfGroup *monsterGroup[5];

class CaveQuakeBullet {

protected:
  pfVec3 dir, pos;
  float speed, lastTime, explodeTime, explodeDuration;
  pfDCS *bulletDcs;
  pfSwitch *bulletSwitch;
  pfSegSet segset;
  int damage, noDamageFlag;
  char *explodeSound;

public:
  CaveQuakeBullet(pfVec3 thePos, pfVec3 theDir, pfNode *bulletNode, pfNode *explodeNode, float theSpeed = 50.0f, int theDamage = 5, float explodeD = 0.5f, char *explodeS = NULL) {
    pos = thePos;
    dir = theDir;
    speed = theSpeed;
    damage = theDamage;
    noDamageFlag = 0;

    explodeDuration = explodeD;

    // set up intersection segments
    segset.activeMask = 1; // which segs are active (bits)
    segset.isectMask = 0xFFFF;
    segset.discFunc = NULL;
    segset.bound = NULL;
    segset.mode = PFTRAV_IS_PRIM|PFTRAV_IS_NORM|PFTRAV_IS_PATH;
    segset.segs[0].length = speed;

    if (bulletNode == NULL)
      bulletNode = new pfDCS;
    if (explodeNode == NULL)
      explodeNode = new pfDCS;

    bulletDcs = new pfDCS;
    //bulletDcs->setScale(0.1f, 0.1f, 0.1f);
    bulletDcs->setScale(0.15f, 0.15f, 0.15f);
    bulletDcs->setTrans(pos[0], pos[1], pos[2]);

    // bullet models point right... rotate into desired heading
    pfDCS *orientDcs = new pfDCS;
    pfMatrix oMat;
    oMat.makeVecRotVec(pfVec3(1.0f, 0.0f, 0.0f), dir);
    orientDcs->setMat(oMat);
    orientDcs->addChild(bulletNode);

    bulletSwitch = new pfSwitch;
    //bulletSwitch->addChild(bulletNode);
    bulletSwitch->addChild(orientDcs);
    bulletSwitch->addChild(explodeNode);
    bulletSwitch->setVal(0);

    bulletDcs->addChild(bulletSwitch);

    if (explodeS != NULL)
      explodeSound = strdup(explodeS);
    else
      explodeSound = NULL;

    lastTime = 0.0f;
  }

  void playerIsect() { segset.isectMask = 0xFFF0; }
  void noDamage() { noDamageFlag = 1; }
  pfDCS *getRoot() { return bulletDcs; }

  void update(pfNode *top) {
    if (bulletSwitch->getVal() == 0) { // bullet is flying
      if (lastTime == 0.0f)
        lastTime = pfGetTime();
      float time = pfGetTime();
      float elapsedTime = time - lastTime;
      if (elapsedTime > 0.2f)
        elapsedTime = 0.2f;

      pfHit **hits[32];
      int isect;
      float d = speed * elapsedTime;

      segset.segs[0].pos.set(pos[0], pos[1], pos[2]);
      segset.segs[0].dir.set(dir[0], dir[1], dir[2]);
      segset.segs[0].length = d;

      isect = top->isect(&segset, hits);

      if (isect) {
        pfVec3 pnt, xpnt;
        pfMatrix xmat;
        pfNode *node = NULL;
        pfPath *path = NULL;
        (*hits[0])->query(PFQHIT_NODE, &node);
        (*hits[0])->query(PFQHIT_PATH, &path);
        (*hits[0])->query(PFQHIT_POINT, pnt.vec);
        (*hits[0])->query(PFQHIT_XFORM, (float*)xmat.mat);
        xpnt.xformPt(pnt, xmat);

        bulletDcs->setTrans(xpnt[PF_X], xpnt[PF_Y], xpnt[PF_Z]);
        bulletSwitch->setVal(1); // explode
        explodeTime = pfGetTime();
      
        float d = 0.0f; 
        if (explodeSound != NULL) {
          d = xpnt.distance(pfVec3(player->X, player->Y, player->Z));
          quakeSound(explodeSound, d/2.0f); // a little louder :)
        }

        if (noDamageFlag) return;

        // rocket - look for monsters, players nearby
        if (damage > 80) {
          if (d < 12.0f) // hit player (half damage)
            player->hit((int) ((120 - d * 10.0f)/2.0f));
          for (int i=0; i<5; i++) {
            if (levelGroup->searchChild(monsterGroup[i]) != -1) {
              int numChildren = monsterGroup[i]->getNumChildren();
              for (int j=0; j<numChildren; j++) {
                CaveQuakeBeing *m = (CaveQuakeBeing *)
                  monsterGroup[i]->getChild(j)->getTravData(PFTRAV_APP);
                float d = xpnt.distance(pfVec3(m->X, m->Y, m->Z));
                if (d < 12.0f)
                    m->hit((int) (120 - d * 10.0f));
              }
            }
          }
        }
        else if (path && node) { // not rocket
          for (int i = path->getNum()-1; i>0; i--) {
            node = (pfNode *) path->get(i);
	    if (node->getName() != NULL)
              if (!strcmp(node->getName(), "monster") || !strcmp(node->getName(), "player"))
                ((CaveQuakeBeing *) node->getTravData(PFTRAV_APP))->hit(damage);
	    
          }
        }
      }
      else { // bullet did not hit
        pos[0] = pos[0] + speed*dir[0]*elapsedTime;
        pos[1] = pos[1] + speed*dir[1]*elapsedTime;
        pos[2] = pos[2] + speed*dir[2]*elapsedTime;
        bulletDcs->setTrans(pos[0], pos[1], pos[2]);
      }

      lastTime = time;
    }
    else { // bullet is not flying
      float time = pfGetTime();
      if (time - explodeTime > explodeDuration) {
        transDcs->removeChild(this->getRoot());
        delete this; // is this ok?
      }
    }
  }
};


static int updateBulletFunc(pfTraverser *trav, void *userData) {
  CaveQuakeBullet *b = (CaveQuakeBullet *) userData;
  b->update(levelGroup);
  return PFTRAV_CONT;
}

#endif
