//====================================================================
//#
//#  CaveQuakeMonster.cxx                              #####     
//#                                                   ##   ##
//#  Author(s): Paul Rajlich                          ##
//#  Email: prajlich@ncsa.uiuc.ed                     ##   ##
//#  Date created:                                     #####
//#
//#  Description: CAVE QUAKE II 3.2
//#
//====================================================================
//#  (c)opyright 1998, 1999  Paul Rajlich, all rights reserved
//====================================================================

#include "CaveQuakeMonster.h"

/*****
int updateMonsterFunc(pfTraverser *trav, void *userData) {
  CaveQuakeMonster *m;
  m = (CaveQuakeMonster *) userData;
  m->update();
  return PFTRAV_CONT;
}
******/

CaveQuakeMonster::CaveQuakeMonster(char *filename) {
  ATTACK = 0; DUCK = 0; PAIN = 0; RUN = 0; STAND = 0;
  WALK = 0; DEATH = 0; FALL = 0;

  ROTSPEED = 0.0f;
  SPEED = 0.0f;
  //WALKSPEED = 4.0f; // defaults, may be overriden in derived classes
  //RUNSPEED = 8.0f;
  WALKSPEED = 4.0f*7.0/5.0;
  RUNSPEED = 8.0f*7.0/5.0;

  monster = pfdLoadFile_md2(filename);
  anim = md2GetNodeAnimation(monster);
  //fprintf(stderr, "NUMBER OF FRAMES: %d\n", anim->getNumFrames());
  initPos();

  enemy = NULL;

  // set up intersection segments
  //segset.activeMask = 3; // which segs are active (bits)
  segset.activeMask = 7; // which segs are active (bits)
  segset.isectMask = 0xFFFF;
  //segset.isectMask = PFBSP_ISECT_MASK_SOLID;
  segset.discFunc = NULL;
  segset.bound = NULL;
  segset.mode = PFTRAV_IS_PRIM|PFTRAV_IS_NORM;
  segset.segs[0].dir.set(0.0f, 0.0f, -1.0f); // down to floor
  segset.segs[0].length = 500.0f;
  segset.segs[1].dir.set(dx, dy, 0.0f); // dir monster is looking
  segset.segs[1].length = 500.0f;

  segset.segs[2].length = 500.0f;

  state = M_NORMAL;
  vSpeed = 0.0;
  forwardHitD = 5.0;

  mood = M_FRIENDLY;
  life = 100;

  lastTime = 0.0f;
  angryTime = 0.0f;
}

void CaveQuakeMonster::reset() {
  life = 100;
  mood = M_FRIENDLY;
  state = M_NORMAL;
  vSpeed = 0.0f;
}  

int CaveQuakeMonster::inSight() {
  // nothing between monster and enemy, within ~10 degrees (cos 10)
  if (lineSightD > (distanceFromEnemy()-1.0f) && cosAngleFromEnemy() > 0.985f)
    return 1;
  else
    return 0;
}
 
// angle between two vectors:  cos t = (a . b) / (|a||b|)
// note: monster dir already normalized.
float CaveQuakeMonster::cosAngleFromEnemy() {
  if (enemy != NULL) {
    float diffX = enemy->X - X;
    float diffY = enemy->Y - Y;
    float xyDist = fsqrt(diffX*diffX + diffY*diffY);
    pfVec3 enemyDir(diffX, diffY, 0.0f);
    pfVec3 monsterDir(dx, dy, 0.0f);
    float cosAngle = monsterDir.dot(enemyDir)/xyDist;
    return cosAngle;
  }
  else 
    return 0.0f;
}

float CaveQuakeMonster::distanceFromEnemy() {
  if (enemy != NULL) {
    pfVec3 monsterPos(X, Y, Z);
    pfVec3 enemyPos(enemy->X, enemy->Y, enemy->Z);
    return monsterPos.distance(enemyPos);
  }
  else return 10.0f;
}

void CaveQuakeMonster::initPos() {
  dcs = new pfDCS();
  setTrans(0.0f, 0.0f, 0.0f);
  setRot(0.0f, 0.0f, 0.0f);
  dcs->setScale(0.12f, 0.12f, 0.12f);
  dcs->addChild(monster);
  dcs->setName("monster"); // for id purposes (like bullet collisions)
  dcs->setTravData(PFTRAV_APP, this);
}
   
void CaveQuakeMonster::setWalkSpeed(float newSpeed) {
  WALKSPEED = newSpeed;
  fprintf(stderr, "new walk speed: %f\n", WALKSPEED);
}

void CaveQuakeMonster::setRunSpeed(float newSpeed) {
  RUNSPEED = newSpeed;
  fprintf(stderr, "new run speed: %f\n", RUNSPEED);
}

void CaveQuakeMonster::setTrans(float x, float y, float z) {
  dcs->setTrans(x, y, z);
  X = x;
  Y = y;
  Z = z;
}

void CaveQuakeMonster::setRot(float h, float p, float r) {
  //H = fmodf(h+360.0f, 360.0f);
  H = h;
  dcs->setRot(H, p, r);
  dy = sin(H*Q_PI/180.0f);
  dx = cos(H*Q_PI/180.0f); 
  //fprintf(stderr, "H: %f\n", H);
}

// UPDATE 
void CaveQuakeMonster::update() {
  float time = pfGetTime();
  float diff = time - lastTime;
  if (diff > 0.2f) 
    diff = 0.2f; // limit

  if (ROTSPEED != 0.0f) {
    H += ROTSPEED*diff;
    setRot(H, 0.0f, 0.0f);
  }
  // only move if collision is more than 3.0 feet away
  if (forwardHitD > 3.0f && distanceFromEnemy() > 4.0f) {
    Y += SPEED*diff*dy;
    X += SPEED*diff*dx;
  }

  if (state == M_FALLING)
    vSpeed -= M_GRAVITY * diff;
 
  Z += vSpeed * diff;

  // update DCS based on current position
  setTrans(X, Y, Z);

  lastTime = time; 
}

void CaveQuakeMonster::random() {

  // only rotate when walking
  if (anim->getCurrAction() == WALK) {
    int i = rand()%20;
    if (i == 0) rotRight();
    else if (i == 1) rotLeft();
    else if (i == 2 || i == 3) rotStop();
  }
  else
    rotStop();

  int action = rand()%40;
  if (action == 10) stand();
  if (action == 20) stand(1);
  if (action == 30 || action == 31) walk();

  if (distanceFromEnemy() < 25.0f)
    sight();
  // if player is spraying bullets...
  else if (distanceFromEnemy() < 60.0f && CAVEBUTTON1) {
    int chance = rand()%10;
    if (chance == 1)
      sight();
  }
}

void CaveQuakeMonster::ai() {
  if (life <= 0) 
    return;

  // after 25s without sight... stop being angry
  if (pfGetTime() > angryTime+25.0f)
    mood = M_FRIENDLY;

  if (mood == M_ANGRY_RAND) {
    int decide = rand()%40;
    if (decide == 1) mood = M_ANGRY;
    if (SPEED > 0.0f) {
      if (decide == 4 || 5) ROTSPEED = -120.0f;
      if (decide == 5 || 6) ROTSPEED = 120.0f;
      if (decide == 7) ROTSPEED = 0.0f;
    }
    else
      rotStop();

    if (cosAngleFromEnemy() < 0.16f) // more than ~90 degrees
      mood = M_ANGRY;
  }

  if (mood == M_ANGRY && enemy != NULL)  {
    int decide = rand()%30;
    if (decide == 1) { // || forwardHitD < 3.0f) { // running into some object
      mood = M_ANGRY_RAND;
      decide = rand()%2;
      if (decide == 0)
        ROTSPEED = -120.0f;
      else
        ROTSPEED = 120.0f;
    }

    // do cross product of monster's heading and desired heading
    float diffX = enemy->X - X;
    float diffY = enemy->Y - Y;
    pfVec3 monsterDir(dx, dy, 0);
    pfVec3 desiredDir(diffX, diffY, 0);
    pfVec3 cross;
    cross.cross(monsterDir, desiredDir);

    // sign of z component is dir to turn
    if (cross[2] > 0.0f)
      ROTSPEED = 120.0f; // rotRight();
    else
      ROTSPEED = -120.0f;

    if (cosAngleFromEnemy() > 0.985f) // within ~10 degrees (cos 10 = 0.9848)
      rotStop();
  }

  if (mood == M_FRIENDLY) 
    random();
}

void CaveQuakeMonster::isect(pfNode *level) {
  if (life <= 0.0)
    return; // no need to isect
  //if (anim->getCurrAction() == STAND || anim->getCurrAction() == STAND+1)
  //  return; // no need to isect
  pfHit **hits[32];
  int isect;

  segset.segs[0].pos.set(X, Y, Z+6.0f);
  segset.segs[1].pos.set(X, Y, Z+6.0f);
  segset.segs[1].dir.set(dx, dy, 0.0f);

  segset.segs[2].pos.set(X, Y, Z+6.0f);
  segset.segs[2].dir.set(enemy->X - X, enemy->Y - Y, enemy->Z - Z);

  isect = level->isect(&segset, hits);

  if (isect) {
    pfVec3 pnt, xpnt;
    pfMatrix xmat;

    (*hits[0])->query(PFQHIT_POINT, pnt.vec);
    (*hits[0])->query(PFQHIT_XFORM, (float*)xmat.mat);
    xpnt.xformPt(pnt, xmat);
    // give him falling and rising
    float dist = xpnt[PF_Z] - Z;
    if (dist < -0.3f)
      state = M_FALLING;
    else {
      state = M_NORMAL;
      vSpeed = 0.0f;
      Z = xpnt[PF_Z];
    }

    (*hits[1])->query(PFQHIT_POINT, pnt.vec);
    (*hits[1])->query(PFQHIT_XFORM, (float*)xmat.mat);
    xpnt.xformPt(pnt, xmat);
    // distance from head to collision
    forwardHitD = xpnt.distance(pfVec3(X, Y, Z+6.0));

    (*hits[2])->query(PFQHIT_POINT, pnt.vec);
    (*hits[2])->query(PFQHIT_XFORM, (float*)xmat.mat);
    xpnt.xformPt(pnt, xmat);
    // distance to enemy or to obstacle inbetween
    lineSightD = xpnt.distance(pfVec3(X, Y, Z+6.0));
  }
}

void CaveQuakeMonster::hit(int damage) {
  life -= damage;

  if (life > 0) {
    if (mood == M_FRIENDLY)
      sight();
    else {
      int i = rand()%3;
      if (i == 1)
        pain(); // more severe pain
      else 
        pain(1);
    }
  }
  //else if (life < -40) reset(); // revive the dead
  else {
    if (anim->getCurrAction() != DEATH)
      death();
    mood = M_DEAD;
    ROTSPEED = 0.0f;   
  }
} 

