//====================================================================
//#
//#  caveQuake.cxx                                     #####     
//#                                                   ##   ##
//#  Author(s): Paul Rajlich                          ##
//#  Email: prajlich@ncsa.uiuc.ed                     ##   ##
//#  Date created:                                     #####
//#
//#  Description: CAVE QUAKE II
//#
//====================================================================
//#  (c)opyright 1998, 1999, 2000  Paul Rajlich, all rights reserved
//====================================================================

#ifdef USE_CAVELIB
#include <pfcave.h>
#else
#include "pfbox.h"
#include "pfcaveAlt.h"
#endif
#include <Performer/pfdu.h>
#include <Performer/pf/pfChannel.h>
#include <Performer/pf/pfLightSource.h>
#include <Performer/pf/pfTraverser.h>
#include <Performer/pf/pfGeode.h>
#include <Performer/pf/pfDCS.h>
#include <Performer/pf/pfSwitch.h>
#include <Performer/pr/pfGeoSet.h>
#include <Performer/pr/pfTexture.h>
#include <Performer/pfutil.h>
#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include <fcntl.h>

// audio
//#include <gl.h> // contains Boolean type vss expects
//#include "vssClient_v2.2.h"
#ifdef USE_VSS
#include "vssClient.h"
#endif
#include "utils.h"

// player
#include "CaveQuakePlayer.h"
#include "CaveQuakeOpponent.h"
#include "CaveQuakeHealth.h"
#include "CaveQuakeArmor.h"

// monsters
#include "CaveQuakeBerserk.h"
#include "CaveQuakeGunner.h"
#include "CaveQuakeTank.h"
#include "CaveQuakeInfantry.h"
#include "CaveQuakeSoldier.h"

// level loader
#include "CaveQuakeBSP.h"


// function protos
static void initSharedVars(int argc, char **argv);
static void createScene(pfChannel *chan, int argc, char **argv);
static void loadLevel(pfChannel *chan, int argc, char **argv); 
pfGeode *makePoster(char *filename);
static void switchLevel(int num, int which=0);
static void moveViper();
static void checkForEvents();
static pfNode *makeMonster(char type, float x, float y, float z);
static int updateMonsterFunc(pfTraverser *trav, void *userData);
static int updateItemFunc(pfTraverser *trav, void *userData);
static void printInfo();


// globals
CaveQuakePlayer *player;
extern float playerStart[4];
extern float playerPlat[2];

// shared vars
int *audioToggle, *audioHandle, *themeToggle;
pfDCS *transDcs, *rotDcs, *levelDcs, *titleDcs, *viperDcs, *dishDcs;
float *eyePos;
pfSwitch *toggleScene;
int *currLevel, *viperToggle;
CaveQuakeBSP *currBsp, *bsps[6];
pfGroup *monsterGroup[5], *itemGroup[5];
pfGroup *gameGroup, *levelGroup;
CaveQuakeOpponent *you;
float *gammaCorrect;
pfNode *laser, *rocket;
pfDCS *explodeDcs;

#ifdef USE_CAVELIB
void initBinOrders(void)
{
    pfChannel* chan;
    pfList*    chanList = pfCAVEChannels();

    pfCAVEMasterChan()->setBinOrder(2,2);
    pfCAVEMasterChan()->setBinOrder(3,3);
//  pfCAVEMasterChan()->setTravMode(PFTRAV_CULL, PFCULL_SORT);

    for(int i=0; i<(chanList->getNum()); i++)
    {
        chan = (pfChannel*)(chanList->get(i));

        chan->setBinOrder(2,2);
        chan->setBinOrder(3,3);
//      chan->setTravMode(PFTRAV_CULL, PFCULL_SORT);
    }
}
#endif

int isEnv(char *var, char *str) {
  char *env = getenv(var);
  if (!env) return 0;
  if (strcmp(env, str)) return 0;
  return 1;
}

int main(int argc, char **argv) {
  
  int posPrint = 0;

  pfInit();
#ifdef USE_CAVELIB
  pfCAVEConfig(&argc, argv, NULL);
#else
  pfMultiprocess(PFMP_DEFAULT);
#endif

  printInfo();
  //pfMultiprocess(PFMP_APPCULLDRAW);

  // 1000 ft.
#ifdef USE_CAVELIB
  CAVEFar = 1000;
#else
  BOXSetFar(1000.0f);
#endif

  // init loader DSOs before pfConfig so that each process has them
  pfdInitConverter("obj");
  pfdInitConverter_md2();
  pfdInitConverter_bsp();
  if (isEnv("CAVE_QUAKE_LIGHTMAPS", "off")) {
    cout << "\n-------- LIGHTMAPS DISABLED --------\n" << endl;
    pfdConverterMode_bsp(PFBSP_USE_LIGHTMAPS, PF_OFF);
  }
  //pfdConverterMode_bsp(PFBSP_RANDOM_LEAF_COLORS, PF_ON);
  //pfFilePath(".:/usr/share/Performer/data"); // fonts
  pfFilePath(".:../fonts"); // fonts

  initSharedVars(argc, argv); 

  // normal pfCAVE config
  pfConfig();
#ifdef USE_CAVELIB
  pfCAVEInitChannels();
#else
  BOXInit();
  BOXSetKeyDownCB(keyDown);
  BOXSetAppCB(NULL);
  BOXSetHeadOffset(0.0f, 4.0f, 0.0f);
#endif

  // middle button prints out player pos
  if (isEnv("CAVE_QUAKE_POS_PRINT", "on"))
    posPrint = 1;

  // show me the stats... this config only works on desktop
#ifdef USE_CAVELIB
  // draw textures, then lightmaps
  initBinOrders();
  if (isEnv("CAVE_QUAKE_STATS", "on"))
    pfCAVEMasterChan()->getFStats()->setClass(PFSTATS_ALL^PFSTATSHW_ENGFXPIPE_FILL, PFSTATS_ON);
#endif

  // intersection testing in seperate process? Also need to do pfMultiProcess()
  //pfIsectFunc(isectFunc);

  // disable theme...
  if (isEnv("CAVE_QUAKE_THEME", "off")) {
    cout << "\n-------- INTRO MUSIC DISABLED --------\n" << endl;
    *themeToggle = 0;
  }

#ifdef USE_VSS
  // connect to audio server (vss2.2)
  if (*audioToggle) {
    if (!BeginSoundServer()){
      printf("UDP connection to sound server failed!\n");
      *audioToggle = 0;
    }
    else {
      char audFilename[150];
      if (argc < 2)
        sprintf(audFilename,"AUD/quake.aud");
      else
        sprintf(audFilename, argv[1]);
      *audioHandle = AUDinit(audFilename);
      printf("AUD file %s opened.\n", audFilename);
    }
  }
  else 
#endif
    fprintf(stderr, "----- AUDIO OFF -----\n");

#ifdef USE_CAVELIB
  createScene(pfCAVEMasterChan(), argc, argv);
#else
  createScene(BOXMasterChan(), argc, argv);
#endif

  rotDcs = new pfDCS();
  transDcs = new pfDCS();
  levelGroup = new pfGroup();

#ifdef USE_CAVELIB
  loadLevel(pfCAVEMasterChan(), argc, argv);
#else
  loadLevel(BOXMasterChan(), argc, argv);
#endif

  //float IOdistance = CAVEConfig->InterocularDistance;
  //cout << "RW: TYNS" << endl;

  if (isEnv("CAVE_QUAKE_VPAIN", "off"))
    player->setPainToggle(0);

  float introTime = pfGetTime();
  while (!CAVEgetbutton(CAVE_ESCKEY)) {
    float time = pfGetTime()-introTime;
    //float time1 = pfGetTime();

    playerPlat[0] = 0; // reset on-platform flag

    // reset at beginning of current level...
    if (CAVEgetbutton(CAVE_RKEY) || (CAVEBUTTON1 && CAVEBUTTON2 && CAVEBUTTON3)) {
      switchLevel(*currLevel);
    }

    float headPos[3], wandPos[3], wandO[3], headO[3];
    CAVEGetPosition (CAVE_HEAD, headPos);
    CAVEGetOrientation (CAVE_WAND, wandO);
    CAVEGetPosition (CAVE_WAND, wandPos);
    CAVEGetOrientation (CAVE_HEAD, headO);

    pfVec3 worldPos; 
    pfMatrix rotM;
    rotM.makeRot(-(player->H), 0.0f, 0.0f, 1.0f);
    worldPos.xformPt(pfVec3(headPos[0], headPos[1], 0.0f), rotM);

    // button 3 returns to directions
    if (CAVEButtonChange(3) == -1) {
      if (toggleScene->getVal() == 0) {
        you->switchModel();
        quakeSound("pMenu", 3.0f);
      }
      toggleScene->setVal(0);
      if (player->getLife() <= 0)
        player->reset();
    }

    // intro...
    if (time < 3.0f)
      titleDcs->setTrans(0.0f, 15.0f-5.0f*time, 6.0f);
    else if (time < 4.0f)
      titleDcs->setTrans(0.0f, (time-3.0f)*5.0f, 6.0f);
    else if ((toggleScene->getVal() == 0) && (CAVEBUTTON1 || CAVEBUTTON2)) {
      toggleScene->setVal(2);
      if (CAVEBUTTON1)
        player->setJoystickNav(1);
      else // button2
        player->setJoystickNav(0);
    }

    pfSync();
#ifdef USE_CAVELIB
    pfCAVEPreFrame();
#endif
    player->isect(rotDcs); // update isect information
    //pfCAVEMasterChan()->drawStats();
    //if (playerPlat[0])
    //  player->Z = playerPlat[1]/10.0f;
#ifdef USE_CAVELIB
    pfFrame();
    pfCAVEPostFrame();
#else
    updateWing();
    updateWanda();
    BOXUpdate();
#endif

    //float time2 = pfGetTime();

    if (toggleScene->getVal() == 2) {

      // update player
      //player->isect(rotDcs); // update isect information
      player->update((qkBsp *) currBsp->bsp);      // move based on user input

      // print stats
      if (CAVEBUTTON2 && posPrint)
        cout << "player pos: " << player->X << " " << player->Y << " "
             << player->Z << " H: " << player->H << " life: "
             << player->getLife() <<endl;

      if (CAVEgetbutton(CAVE_XKEY)) 
        pfPrint(rotDcs, PFTRAV_SELF | PFTRAV_DESCEND, PFPRINT_VB_ON, stdout);

      // update global info
      rotDcs->setRot(player->H, player->P, player->R);
      transDcs->setTrans(-player->X, -player->Y, -player->Z);

      // PAUL!!! added head offset (worldPos)
      eyePos[0] = (player->X+worldPos[0]) * 10.0f;
      eyePos[1] = (player->Y+worldPos[1]) * 10.0f;
      eyePos[2] = (player->Z + headPos[2]) * 10.0f; // experiment
      //eyePos[2] = (player->Z + 6.0f) * 10.0f; // player->Z is floor height
      //eyePos[2] = player->Z * 10.0f;
      // PAUL - look into this...

      if (*viperToggle) 
        moveViper();

      checkForEvents(); // level switches, win, etc...
      //cout << "etime1: " << etime1 << " etime2: " << pfGetTime()-time2 << endl;
    }
  }

#ifdef USE_CAVELIB
  CAVEHalt();
#else
  BOXExit();
#endif
  pfExit();

#ifdef USE_VSSS
  if (*audioToggle) {
    AUDterminate(*audioHandle);
    EndSoundServer();
  }
#endif

  return 0;
}

void initSharedVars(int argc, char **argv) {
  currLevel = (int *) pfMalloc (sizeof(int), pfGetSharedArena());
  *currLevel = 1;
  viperToggle = (int *) pfMalloc (sizeof(int), pfGetSharedArena());
  *viperToggle = 0;

  audioHandle = (int *) pfMalloc (sizeof(int), pfGetSharedArena());
  audioToggle = (int *) pfMalloc (sizeof(int), pfGetSharedArena());
  *audioToggle = 1;

  themeToggle = (int *) pfMalloc (sizeof(int), pfGetSharedArena());
  *themeToggle = 1;

  eyePos = (float *) pfMalloc (3*sizeof(float), pfGetSharedArena());
  eyePos[0] = 0.0;  eyePos[1] = 0.0;  eyePos[2] = 5.0;

  gammaCorrect = (float *) pfMalloc (sizeof(float), pfGetSharedArena());
  *gammaCorrect = 1.4f;
  if (argc > 4)
    *gammaCorrect = atof(argv[4]);
}

static int updateItemFunc(pfTraverser *trav, void *userData) {
  CaveQuakeItem *i = (CaveQuakeItem *) userData;
  i->update();
  return PFTRAV_CONT;
}

static int updateMonsterFunc(pfTraverser *trav, void *userData) {

  CaveQuakeMonster *m = (CaveQuakeMonster *) userData;

  quakeBsp *bsp = currBsp->bsp;
  float eye[3];
  eye[0] = m->X*10.0f;
  eye[1] = m->Y*10.0f;
  eye[2] = (m->Z+5.0f)*10.0f;
  //cout << "monster " << i << " eye: " << eye[0] << " " << eye[1] <<
  //        " " << eye[2] << endl;
  int result = qkFindCurCluster(eye, (qkBsp *) bsp);

  if (result == 0) { // monster not visible
    m->getRoot()->setTravMask(PFTRAV_DRAW, 0x0, PFTRAV_DESCEND, PF_SET);
    m->getRoot()->setTravMask(PFTRAV_ISECT, 0x0, PFTRAV_DESCEND, PF_SET); 
  }
  else if (result == -1) { // monster in negative cluster
    //cout << "restoring monster pos" << endl;
    m->restorePos(); // restore previous position
    m->getRoot()->setTravMask(PFTRAV_DRAW, 0xFFFFFFFF, PFTRAV_DESCEND, PF_SET);
    m->getRoot()->setTravMask(PFTRAV_ISECT, 0xFFFF, PFTRAV_DESCEND, PF_SET); 
    m->isect(levelDcs);
    //m->isect(levelGroup); // other monsters also
  }
  else { // monster is visible
    //cout << "#";
    m->savePos(); // save curr position
    m->getRoot()->setTravMask(PFTRAV_DRAW, 0xFFFFFFFF, PFTRAV_DESCEND, PF_SET);
    m->getRoot()->setTravMask(PFTRAV_ISECT, 0xFFFF, PFTRAV_DESCEND, PF_SET); 
    m->ai();
    m->update();
    m->isect(levelDcs);
    //m->isect(levelGroup);
  }

  return PFTRAV_CONT;
}
  

static void loadLevel(pfChannel *chan, int argc, char **argv) {
  pfScene *scene = chan->getScene();

  player = new CaveQuakePlayer();
  if (argc > 2)
    player->setRotSpeed(player->getRotSpeed() * atof(argv[2]));

  if (isEnv("CAVE_QUAKE_STRAFE", "off")) {
    cout << "\n-------- STRAFING DISABLED --------\n" << endl;
    player->setStrafe(0);
  }
  if (isEnv("CAVE_QUAKE_JOYSTICK", "off")) {
    cout << "\n-------- JOYSTICK DISABLED --------\n" << endl;
    player->setNoJoystick(1);
  }

  //scene->addChild(player->getStatsDcs());
  levelGroup->addChild(player->makeGeom());

  // scale level (no need to center)
  levelDcs = new pfDCS();
  levelDcs->setScale(0.1f, 0.1f, 0.1f);
  bsps[0] = NULL;
  bsps[1] = NULL;
  bsps[2] = NULL;
  bsps[3] = NULL;
  bsps[4] = NULL;
  bsps[5] = NULL;
  // preload...
  //bsps[0] = new CaveQuakeBSP("maps/demo1.bsp"); 
  //bsps[1] = new CaveQuakeBSP("maps/demo2.bsp"); 
  //bsps[2] = new CaveQuakeBSP("maps/demo3.bsp"); 
  monsterGroup[0] = NULL;
  monsterGroup[1] = NULL;
  monsterGroup[2] = NULL;
  monsterGroup[3] = NULL;
  monsterGroup[4] = NULL;

  // init items
  for (int i=0; i<5; i++)
    itemGroup[i] = NULL;

  if (argc < 4)
    switchLevel(1);
  else
    switchLevel(atoi(argv[3]));

  /*** 
  pfNode *level = currBsp->level;
  pfSphere* bounds = new pfSphere;
  level->getBound(bounds);
  fprintf(stderr, "scaleFactor: 0.1\n");
  fprintf(stderr, "center: %f %f %f\n", bounds->center[0], bounds->center[1],
          bounds->center[2]);
  ****/

  // note: want to rotate around user, not around center of level
  //   so, move world so that user is in correct spot, then rotate
  //rotDcs = new pfDCS();
  //transDcs = new pfDCS();
  rotDcs->setRot(player->H, player->P, player->R);
  transDcs->setTrans(-player->X, -player->Y, -player->Z);

  rotDcs->addChild(transDcs);

  // everything used to be under transDcs... now under levelGroup for isect
  transDcs->addChild(levelGroup);
  levelGroup->addChild(levelDcs);

  // pfEarthSky doesn't work right in the CAVE. :(
  // Make big skybox instead... center is -100 100 -100
  pfDCS *skyLeft = new pfDCS;
  skyLeft->setScale(80.0f, 80.0f, 80.0f);
  skyLeft->setRot(90.0f, 0.0f, 0.0f);
  skyLeft->setTrans(-400.0f, 100.0f, -100.0f);
  skyLeft->addChild(makePoster("imgs/sky1lf.rgb"));

  pfDCS *skyTop = new pfDCS;
  skyTop->setScale(80.0f, 80.0f, 80.0f);
  skyTop->setRot(-90.0f, 90.0f, 0.0f);
  skyTop->setTrans(-100.0f, 100.0f, 300.0f);
  skyTop->addChild(makePoster("imgs/sky1up.rgb"));

  pfDCS *skyRight = new pfDCS;
  skyRight->setScale(80.0f, 80.0f, 80.0f);
  skyRight->setRot(-90.0f, 0.0f, 0.0f);
  skyRight->setTrans(300.0f, 100.0f, -100.0f);
  skyRight->addChild(makePoster("imgs/sky1rt.rgb"));

  pfDCS *skyFront = new pfDCS;
  skyFront->setScale(80.0f, 80.0f, 80.0f);
  skyFront->setRot(180.0f, 0.0f, 0.0f);
  skyFront->setTrans(-100.0f, -300.0f, -100.0f);
  skyFront->addChild(makePoster("imgs/sky1ft.rgb"));

  pfDCS *skyBack = new pfDCS;
  skyBack->setScale(80.0f, 80.0f, 80.0f);
  skyBack->setRot(0.0f, 0.0f, 0.0f);
  skyBack->setTrans(-100.0f, 500.0f, -100.0f);
  skyBack->addChild(makePoster("imgs/sky1bk.rgb"));

  levelGroup->addChild(skyLeft);
  levelGroup->addChild(skyTop);
  levelGroup->addChild(skyRight);
  levelGroup->addChild(skyFront);
  levelGroup->addChild(skyBack);

  //pfGroup *gameGroup = new pfGroup();
  gameGroup = new pfGroup();
  gameGroup->addChild(rotDcs);
  gameGroup->addChild(player->getGunDcs());
  gameGroup->addChild(player->getStatsDcs());
  //gameGroup->addChild(player->getExplodeDcs());

  // add level and switch to it (after intro is over)
  toggleScene->addChild(gameGroup);
}

static void createScene(pfChannel *chan, int argc, char **argv) {
  pfScene *scene;
  pfGeoState *gstate;
  pfLightSource *light;
  scene = new pfScene;
  gstate = new pfGeoState;
  gstate->setMode(PFSTATE_ENLIGHTING, PF_ON);
//  gstate->setMode(PFSTATE_CULLFACE, PFCF_OFF);
  scene->setGState(gstate);

  light = new pfLightSource;
  light->setPos(0.0f, -1.0f, 0.0f, 0.0f);
  light->setColor(PFLT_DIFFUSE, 0.8, 0.8, 0.8);
  light->setAmbient(0.1, 0.1, 0.1);
  scene->addChild(light);

  pfDCS *introDcs = new pfDCS();
  titleDcs = new pfDCS();
  titleDcs->setTrans(0.0f, 55.0f, 6.0f);
  pfNode *title = makePoster("models/monsters/soldier/.title");
  titleDcs->addChild(title);
  introDcs->addChild(titleDcs);

  you = new CaveQuakeOpponent;
  you->setTrans(-6.0f, 4.0f, 0.0f);
  you->stand();
  introDcs->addChild(you->getRoot());

  pfDCS *youDcs = new pfDCS();
  youDcs->setScale(0.2f, 0.2f, 0.2f);
  youDcs->setRot(45.0f, 0.0f, 0.0f);
  youDcs->setTrans(-7.0f, 5.0f, 6.2f);
  youDcs->addChild(makePoster("imgs/you.rgb"));
  introDcs->addChild(youDcs);

  //skullDcs = new pfDCS();
  //skullDcs->setScale(0.1f, 0.1f, 0.1f);
  //skullDcs->setRot(0.0f, 0.0f, 0.0f);
  //skullDcs->setTrans(0.0f, 0.0f, 1.0f);
  //skullDcs->addChild(pfdLoadFile_md2("models/items/invulner/tris.md2"));
  //introDcs->addChild(skullDcs); 

  pfDCS *floorDcs = new pfDCS();
  floorDcs->setRot(0.0f, 270.0f, 0.0f);
  floorDcs->setTrans(0.0f, 0.0f, 0.0f);
  floorDcs->addChild(makePoster("imgs/qlogo.rgb"));
  introDcs->addChild(floorDcs);

  pfGroup *winGroup = new pfGroup;
  
  pfDCS *winDcs = new pfDCS;
  winDcs->setScale(0.4f, 0.4f, 0.4f);
  winDcs->setTrans(-2.25f, 5.0f, 5.0f);
  pfNode *win = makePoster("imgs/win.rgb");
  winDcs->addChild(win);
  winGroup->addChild(winDcs);

  pfDCS *paulDcs = new pfDCS;
  paulDcs->setScale(0.4f, 0.4f, 0.4f);
  paulDcs->setTrans(2.25f, 5.0f, 5.0f);
  //paulDcs->setTrans(11.25f, 0.01f, 0.0f);
  pfNode *paul = makePoster("imgs/DESK1c.rgb");
  paulDcs->addChild(paul);
  //winDcs->addChild(paulDcs);
  winGroup->addChild(paulDcs);

  //pfDCS *meDcs = new pfDCS;
  //meDcs->setScale(3.6f, 3.6f, 3.6f);
  //meDcs->setRot(30.0f, 90.0f, 0.0f);
  //meDcs->setTrans(-2.5f, 3.0f, 0.0f);
  //meDcs->addChild(pfdLoadFile("paul/paulUnix.obj"));
  //winGroup->addChild(meDcs);
  winGroup->addChild(floorDcs);

  viperDcs = new pfDCS;
  viperDcs->setScale(0.1f, 0.1f, 0.1f);
  viperDcs->setTrans(0.0f, 20.0f, -3.0f); 
  viperDcs->setRot(240.0f, 0.0f, 0.0f);
  pfNode *viper = pfdLoadFile_md2("models/ships/bigviper/tris.md2");
  viperDcs->addChild(viper);

  toggleScene = new pfSwitch();
  toggleScene->addChild(introDcs);
  toggleScene->addChild(winGroup);
  toggleScene->setVal(0);

  scene->addChild(toggleScene);
  chan->setScene(scene);

  // dish... used in level 3
  dishDcs = new pfDCS;
  dishDcs->setScale(0.15f, 0.15f, 0.15f);
  dishDcs->setTrans(-65.3f, -2.0f, -23.3f);
  pfNode *dish = pfdLoadFile_md2("models/objects/satellite/tris.md2");
  dishDcs->addChild(dish);

  // laser, rocket, explosion
  laser = pfdLoadFile_md2("models/objects/laser/tris.md2");
  rocket = pfdLoadFile_md2("models/objects/rocket/tris.md2");
  explodeDcs = new pfDCS();
  explodeDcs->setScale(5.0f, 5.0f, 5.0f);
  pfNode *explode = pfdLoadFile_md2("models/objects/explode/tris.md2");
  explodeDcs->addChild(explode);

  // add dish to intro screen (to show what to look for)
  //pfDCS *introDishDcs = new pfDCS;
  //introDishDcs->setScale(0.04f, 0.04f, 0.04f);
  //introDishDcs->setRot(180.0f, 0.0f, 0.0f);
  //introDishDcs->setTrans(8.0f, 4.0f, 0.0f);
  //introDishDcs->addChild(dish);
  //introDcs->addChild(introDishDcs);
}

// load texture onto 10x10 square centered at (0,0).
pfGeode* makePoster(char *filename) {

  pfGeode *title = new pfGeode;
  title->setName("title");

  int nVertices = 4;
  pfVec3 *v = (pfVec3 *) pfMalloc(nVertices*sizeof(pfVec3), pfGetSharedArena());
  v[0].set(5.0f, 0.0f, -5.0f);
  v[1].set(5.0f, 0.0f, 5.0f);
  v[2].set(-5.0f, 0.0f, -5.0f);
  v[3].set(-5.0f, 0.0f, 5.0f);

  int *lengths = (int *) pfMalloc(sizeof(int), pfGetSharedArena());
  lengths[0] = nVertices;
  pfGeoSet *gset = new pfGeoSet;
  gset->setPrimType(PFGS_TRISTRIPS);
  gset->setNumPrims(1);
  gset->setPrimLengths(lengths);
  gset->setAttr(PFGS_COORD3, PFGS_PER_VERTEX, v, NULL);
  pfVec4 *color = (pfVec4 *) pfMalloc(sizeof(pfVec4), pfGetSharedArena());
  color->set(1.0f, 1.0f, 1.0f, 1.0f);
  gset->setAttr(PFGS_COLOR4, PFGS_OVERALL, color, NULL);

  pfTexEnv *texEnv = new pfTexEnv;
  texEnv->setMode(PFTE_MODULATE);
  pfTexture *tex = new pfTexture;
  tex->loadFile(filename);
  uint *i;
  int nc, sx, sy, sz;
  tex->getImage(&i, &nc, &sx, &sy, &sz);

  pfVec2 *tcoords = (pfVec2*) pfMalloc(4*sizeof(pfVec2), pfGetSharedArena());
  tcoords[0].set(0.0f, 0.0f);
  tcoords[1].set(0.0f, 1.0f);
  tcoords[2].set(-1.0f, 0.0f);
  tcoords[3].set(-1.0f, 1.0f);

  pfGeoState *gstate = new pfGeoState;
  gstate->setMode(PFSTATE_ENLIGHTING, PF_OFF);
  gstate->setMode(PFSTATE_ENTEXTURE, PF_ON);
  gstate->setAttr(PFSTATE_TEXTURE, tex);
  gstate->setAttr(PFSTATE_TEXENV, texEnv);

  gset->setAttr(PFGS_TEXCOORD2, PFGS_PER_VERTEX, tcoords, NULL);
  gset->setGState(gstate);

  title->addGSet(gset);
  
  return title;
}

// utility function for setting up monsters. Could move this into
// a monster constructor? (no, lot of this is application specific).
static pfNode *makeMonster(char type, float x, float y, float z) {

  CaveQuakeMonster *m;
  if (type == 's') m = new CaveQuakeSoldier();
  if (type == 'i') m = new CaveQuakeInfantry();
  if (type == 'b') m = new CaveQuakeBerserk();
  if (type == 'g') m = new CaveQuakeGunner();
  if (type == 't') m = new CaveQuakeTank();

  m->setTrans(x, y, z);
  m->savePos();
  m->reset();
  m->stand();
  //m->walk();
  m->setEnemy(player);                   //pre             //post
  m->getRoot()->setTravFuncs(PFTRAV_APP, NULL, updateMonsterFunc);
  m->getRoot()->setTravData(PFTRAV_APP, m);
  // note: post callback so that player can affect (ie hurt) monster
  //       before the next frame.

  return m->getRoot();
}

static pfNode *makeItem(char type, float x, float y, float z) {
  CaveQuakeItem *i;
  if (type == 'h') i = new CaveQuakeHealth(x, y, z, 60);
  if (type == 'a') i = new CaveQuakeArmor(x, y, z, 60);

  i->getRoot()->setTravFuncs(PFTRAV_APP, NULL, updateItemFunc);
  i->getRoot()->setTravData(PFTRAV_APP, i);
 
  return i->getRoot();
}

static void switchLevel(int num, int which) {
#ifdef USE_VSS
  if (*audioToggle && bsps[num-1] == NULL && *themeToggle)
    AUDupdate(*audioHandle, "theme", 0, NULL);
#endif

  // remove current level...
  pfNode *level = levelDcs->getChild(0);
  if (level)
    levelDcs->removeChild(level);

  if (num == 1) {
    if (bsps[0] == NULL)
      bsps[0] = new CaveQuakeBSP("maps/demo1.bsp", *gammaCorrect);
    currBsp = bsps[0];
    pfNode *level1 = currBsp->level;
    levelDcs->addChild(level1);

    if (which == 1) {
      player->X = -172.1f;
      player->Y = 152.9f;
      player->Z = 9.8f;
      player->H = 90.0f;
    }
    else { 
      player->X = 15.7; 
      player->Y = -36.2;
      //player->Z = -1.5;
      player->Z = 0.0;
      player->H = 0.0f;
      // SIGGRAPH
      player->X = -65.8;
      player->Y = 20.8;
      player->Z = -4.5;
      player->H = -35.7; 
    }

    if (itemGroup[0] == NULL) {
      itemGroup[0] = new pfGroup;

      itemGroup[0]->addChild(makeItem('h', 15.7f, 6.4f, 0.0f));
      itemGroup[0]->addChild(makeItem('h', -92.6f, 25.0f, -4.6f));
      itemGroup[0]->addChild(makeItem('h', -38.9f, 57.1f, -12.9f));
      itemGroup[0]->addChild(makeItem('h', -2.8f, 73.3f, -6.2f));
      itemGroup[0]->addChild(makeItem('h', -120.0f, 177.8f, -4.8f));
      itemGroup[0]->addChild(makeItem('h', -199.8f, 132.4f, -4.8f));

      itemGroup[0]->addChild(makeItem('a', -38.4f, -54.0f, -10.4f));
      itemGroup[0]->addChild(makeItem('a', -7.9f, 142.6f, -12.6f));
      itemGroup[0]->addChild(makeItem('a', -61.4f, 86.5f, -5.0f));
      itemGroup[0]->addChild(makeItem('a', -163.4f, 149.9f, -3.1f));
    }
    levelGroup->removeChild(itemGroup[0]);
    levelGroup->removeChild(itemGroup[1]);
    levelGroup->removeChild(itemGroup[2]);
    levelGroup->removeChild(itemGroup[3]);
    levelGroup->removeChild(itemGroup[4]);
    levelGroup->addChild(itemGroup[0]);

    if (monsterGroup[0] == NULL) {
      monsterGroup[0] = new pfGroup;
  
      // soldiers
      monsterGroup[0]->addChild(makeMonster('s', 10.5, 2.0, 0.0));
      monsterGroup[0]->addChild(makeMonster('s', -72.5, 32.7, -4.0));
      monsterGroup[0]->addChild(makeMonster('s', -80.5f, 34.2f, -4.5f));
      monsterGroup[0]->addChild(makeMonster('s', -9.5f, 96.3f, -18.3f));
      //monsterGroup[0]->addChild(makeMonster('s', -63.3f, 124.0f, -29.4f));
      monsterGroup[0]->addChild(makeMonster('s', -128.5f, 171.3f, -4.5f));
      monsterGroup[0]->addChild(makeMonster('s', -134.1f, 132.7f, -4.5f));
      monsterGroup[0]->addChild(makeMonster('s', -60.5, 67.14, -4.0));

      // infantry
      monsterGroup[0]->addChild(makeMonster('i', -13.0, 124.0, -25.0));
      monsterGroup[0]->addChild(makeMonster('i', 47.4f, -47.3f, -14.1f));
      monsterGroup[0]->addChild(makeMonster('i', 84.0f, 4.7f, -19.3f));
      //monsterGroup[0]->addChild(makeMonster('i', -91.3f, 128.6f, -4.7f));

      // berserks
      monsterGroup[0]->addChild(makeMonster('b', -66.3f, 72.3f, -4.7f));
      //monsterGroup[0]->addChild(makeMonster('b', -136.9f, 165.3f, -4.7f));
      monsterGroup[0]->addChild(makeMonster('b', -13.5, 76.0, -10.0));
      //monsterGroup[0]->addChild(makeMonster('b', -31.8f, 2.7f, -4.9f));

      // gunners
      monsterGroup[0]->addChild(makeMonster('g', -181.9f, 128.4f, 9.3f));
      monsterGroup[0]->addChild(makeMonster('g', -82.5f, 75.3f, -4.7f));
      monsterGroup[0]->addChild(makeMonster('g', 78.2f, -0.4f, -5.6f));

      // tanks
      monsterGroup[0]->addChild(makeMonster('t', -136.9f, 160.3f, -4.7f));
    }
    // switch monster groups
    levelGroup->removeChild(monsterGroup[0]);
    levelGroup->removeChild(monsterGroup[1]);
    levelGroup->removeChild(monsterGroup[2]);
    levelGroup->removeChild(monsterGroup[3]);
    levelGroup->removeChild(monsterGroup[4]);
    levelGroup->addChild(monsterGroup[0]);

    // remove dish! 
    levelGroup->removeChild(dishDcs);
  } 
  else if (num == 2) {
    if (bsps[1] == NULL)
      bsps[1] = new CaveQuakeBSP("maps/demo2.bsp", *gammaCorrect);
    //pfNode *level2 = pfdLoadFile_bsp("maps/demo2.bsp");
    currBsp = bsps[1];
    pfNode *level2 = currBsp->level;
    levelDcs->addChild(level2);
    if (which == 1) {
      player->X = 16.0f;
      player->Y = -79.1f;
      player->Z = -1.7f;
    } 
    else { // normal entrance
      player->X = 82.4f; // was 69.4
      player->Y = 230.4f; // was 230.4
      player->Z = -25.5f;
      player->H = -90.0f;
    }

    if (itemGroup[1] == NULL) {
      itemGroup[1] = new pfGroup;
      itemGroup[1]->addChild(makeItem('h', -14.8f, 176.7f, -19.4f));
      itemGroup[1]->addChild(makeItem('h', 55.0f, 122.0f, -19.1f));
      itemGroup[1]->addChild(makeItem('h', 58.4f, 179.4f, 0.0f));
      itemGroup[1]->addChild(makeItem('h', 10.1f, 53.2f, 0.0f));
      itemGroup[1]->addChild(makeItem('h', -91.6f, -6.9f, -16.2f));
      itemGroup[1]->addChild(makeItem('h', 44.1f, -69.7f, -1.3f));
      itemGroup[1]->addChild(makeItem('h', 13.9f, -95.5f, -1.8f));
      itemGroup[1]->addChild(makeItem('a', 28.1f, 193.6f, -19.1f));
      itemGroup[1]->addChild(makeItem('a', 28.4f, 224.0f, 1.0f));
      itemGroup[1]->addChild(makeItem('a', 48.8f, 39.8f, -16.1f));
      itemGroup[1]->addChild(makeItem('a', 6.5f, -32.5f, -1.8f));
      itemGroup[1]->addChild(makeItem('a', 53.8f, -174.9f, 0.0f));
    }
    levelGroup->removeChild(itemGroup[0]);
    levelGroup->removeChild(itemGroup[1]);
    levelGroup->removeChild(itemGroup[2]);
    levelGroup->removeChild(itemGroup[3]);
    levelGroup->removeChild(itemGroup[4]);
    levelGroup->addChild(itemGroup[1]);

    if (monsterGroup[1] == NULL) {

      monsterGroup[1] = new pfGroup();

      // soldiers
      monsterGroup[1]->addChild(makeMonster('s', 42.0f, 238.0f, -25.4f));
      monsterGroup[1]->addChild(makeMonster('s', 12.7f, 229.6f, -21.4f));
      monsterGroup[1]->addChild(makeMonster('s', -19.4f, 189.2f, -19.3f));
      monsterGroup[1]->addChild(makeMonster('s', 12.8f, 136.4f, -19.3f));

      // infantry
      monsterGroup[1]->addChild(makeMonster('i', 11.4f, 103.8f, -19.4f));
      monsterGroup[1]->addChild(makeMonster('i', 48.7f, 8.8f, 0.2f));
      monsterGroup[1]->addChild(makeMonster('i', 7.8f, -27.7f, -1.7f));
      monsterGroup[1]->addChild(makeMonster('i', 73.5f, -129.5f, 0.14f));

      // berserks
      monsterGroup[1]->addChild(makeMonster('b', 41.7f, -106.3f, -0.1f));
      monsterGroup[1]->addChild(makeMonster('b', 30.4f, -130.3f, 0.02f));
      monsterGroup[1]->addChild(makeMonster('b', 19.6f, -75.4f, -1.8f));

      // gunners
      monsterGroup[1]->addChild(makeMonster('g', -57.4f, 12.1f, -17.2f));
      monsterGroup[1]->addChild(makeMonster('g', 49.4f, 139.9f, -19.4f));
      monsterGroup[1]->addChild(makeMonster('g', 9.8f, -22.9f, -1.5f));
      monsterGroup[1]->addChild(makeMonster('g', 39.7f, 127.5f, -19.3f));

      // tanks
      monsterGroup[1]->addChild(makeMonster('t', 16.9f, -177.5f, 0.02f));
      monsterGroup[1]->addChild(makeMonster('t', 23.0f, -76.9f, -1.73f));
    }

    // switch monster groups
    levelGroup->removeChild(monsterGroup[0]);
    levelGroup->removeChild(monsterGroup[1]);
    levelGroup->removeChild(monsterGroup[2]);
    levelGroup->removeChild(monsterGroup[3]);
    levelGroup->removeChild(monsterGroup[4]);
    levelGroup->addChild(monsterGroup[1]);
  
    // remove dish!
    levelGroup->removeChild(dishDcs);
  }
  else if (num == 3) {
    if (bsps[2] == NULL)
      bsps[2] = new CaveQuakeBSP("maps/demo3.bsp", *gammaCorrect);
    currBsp = bsps[2];
    pfNode *level3 = currBsp->level;
    //pfNode *level3 = pfdLoadFile_bsp("maps/demo3.bsp");
    levelDcs->addChild(level3);
    if (which == 1) {
      player->X = 170.4f;
      player->Y = 140.7f;
      player->Z = -83.3f;
      player->H = -90.0f;
    }
    else {
      player->X = 237.5f;
      player->Y = 89.7f;
      player->Z = -32.9f;
      player->H = 270.0f;
      // win area
      //player->X = -48.7f;
      //player->Y = -70.7f;
      //player->Z = -30.5f;
    }

    if (itemGroup[2] == NULL) {
      itemGroup[2] = new pfGroup;
      itemGroup[2]->addChild(makeItem('h', 174.7f, 77.0f, -48.8f));
      itemGroup[2]->addChild(makeItem('h', 115.9f, 155.4f, -83.0f));
      itemGroup[2]->addChild(makeItem('h', 41.7f, 37.2f, -92.7f));
      itemGroup[2]->addChild(makeItem('h', 110.1f, -97.1f, -51.0f));
      itemGroup[2]->addChild(makeItem('h', 115.0f, -10.5f, -51.0f));
      itemGroup[2]->addChild(makeItem('h', 8.4f, -51.5f, -25.6f));
      itemGroup[2]->addChild(makeItem('h', 5.4f, -51.5f, -25.6f));
      itemGroup[2]->addChild(makeItem('h', -74.0f, -45.0f, -30.4f));
      itemGroup[2]->addChild(makeItem('a', 182.4f, 104.9f, -48.4f));
      itemGroup[2]->addChild(makeItem('a', 111.3f, 71.7f, -83.0f));
      itemGroup[2]->addChild(makeItem('a', 29.0f, 72.9f, -56.0f));
      itemGroup[2]->addChild(makeItem('a', 73.6f, -3.8f, -42.4f));
    }
 
    levelGroup->removeChild(itemGroup[0]);
    levelGroup->removeChild(itemGroup[1]);
    levelGroup->removeChild(itemGroup[2]);
    levelGroup->removeChild(itemGroup[3]);
    levelGroup->removeChild(itemGroup[4]);
    levelGroup->addChild(itemGroup[2]);

    if (monsterGroup[2] == NULL) {

      monsterGroup[2] = new pfGroup;

      // soldiers
      monsterGroup[2]->addChild(makeMonster('s', 37.2f, 79.5f, -82.9f));
      monsterGroup[2]->addChild(makeMonster('s', 12.2f, 71.9f, -76.9f));
      monsterGroup[2]->addChild(makeMonster('s', 24.7f, -32.9f, -51.5f));
      monsterGroup[2]->addChild(makeMonster('s', 106.0f, -64.2f, -51.5f));

      // infantry
      monsterGroup[2]->addChild(makeMonster('i', 107.8f, -85.2f, -51.5f));
      monsterGroup[2]->addChild(makeMonster('i', 143.6f, -2.1f, -51.0f));
      monsterGroup[2]->addChild(makeMonster('i', 158.8f, 14.3f, -51.3f));
      monsterGroup[2]->addChild(makeMonster('i', 76.3f, 0.8f, -44.9f));
  
      // berserks
      monsterGroup[2]->addChild(makeMonster('b', 91.5f, 6.9f, -44.9f));
      monsterGroup[2]->addChild(makeMonster('b', 83.7f, 22.9f, -44.9f));
      monsterGroup[2]->addChild(makeMonster('b', 177.1f, 59.1f, -48.6f));
      monsterGroup[2]->addChild(makeMonster('b', -47.9f, -56.2f, -30.4f)); //w

      // gunners 
      monsterGroup[2]->addChild(makeMonster('g', -43.4f, -49.8f, -30.4f)); //w
      monsterGroup[2]->addChild(makeMonster('g', 197.9f, 59.3f, -48.6f));
      monsterGroup[2]->addChild(makeMonster('g', 191.4f, 79.7f, -48.7f));
      monsterGroup[2]->addChild(makeMonster('g', 42.6f, -36.9f, -51.5f));

      // tanks
      monsterGroup[2]->addChild(makeMonster('t', 91.5f, 9.0f, -44.9f));
      monsterGroup[2]->addChild(makeMonster('t', -47.6, -65.4f, -30.4f)); //w
   
    }
    // switch monster groups
    levelGroup->removeChild(monsterGroup[0]);
    levelGroup->removeChild(monsterGroup[1]);
    levelGroup->removeChild(monsterGroup[2]);
    levelGroup->removeChild(monsterGroup[3]);
    levelGroup->removeChild(monsterGroup[4]);
    levelGroup->addChild(monsterGroup[2]);
  
    // add Dish!...
    levelGroup->removeChild(dishDcs);
    levelGroup->addChild(dishDcs);
  }
  else if (num == 4) {
    if (bsps[3] == NULL)
      bsps[3] = new CaveQuakeBSP("maps/Beta2DM.bsp", *gammaCorrect);
    currBsp = bsps[3];
    pfNode *level4 = currBsp->level;
    levelDcs->addChild(level4);
    player->X = 60.0f;
    player->Y = -12.0f;
    player->Z = 3.0f;
    player->H = 250.0f;
    if (itemGroup[3] == NULL) {
      itemGroup[3] = new pfGroup();
      itemGroup[3]->addChild(makeItem('h', 17.1f, -41.1f, -14.5f));
      itemGroup[3]->addChild(makeItem('h', 36.8f, -6.8f, -14.5f));
      itemGroup[3]->addChild(makeItem('h', -15.6f, 71.0f, -14.5f));
      itemGroup[3]->addChild(makeItem('h', -20.9f, 71.0f, -14.5f));
      itemGroup[3]->addChild(makeItem('h', 79.6f, 24.0f, 16.3f));
      itemGroup[3]->addChild(makeItem('h', 52.7f, 35.7f, 31.9f));
      itemGroup[3]->addChild(makeItem('h', -23.4f, 26.1f, 16.2f));
      itemGroup[3]->addChild(makeItem('h', 25.2f, 25.3f, 16.2f));
      itemGroup[3]->addChild(makeItem('a', 101.0f, -38.0f, 16.1f));
      itemGroup[3]->addChild(makeItem('a', -34.0f, -33.1f, 31.9f));
      itemGroup[3]->addChild(makeItem('a', 58.5f, 19.5f, 15.8f));
      itemGroup[3]->addChild(makeItem('a', -39.6f, 78.7f, 16.1f));
    }
    levelGroup->removeChild(itemGroup[0]);
    levelGroup->removeChild(itemGroup[1]);
    levelGroup->removeChild(itemGroup[2]);
    levelGroup->removeChild(itemGroup[3]);
    levelGroup->removeChild(itemGroup[4]);
    levelGroup->addChild(itemGroup[3]);
    if (monsterGroup[3] == NULL) {
      monsterGroup[3] = new pfGroup;
      monsterGroup[3]->addChild(makeMonster('s', 49.7f, -37.0f, -14.4f));
      monsterGroup[3]->addChild(makeMonster('b', 99.3f, 4.0f, 16.0f));
      monsterGroup[3]->addChild(makeMonster('s', 132.0f, -7.0f, 16.2f));
      monsterGroup[3]->addChild(makeMonster('t', 28.0f, -68.0f, 31.8f));
    }
    levelGroup->removeChild(monsterGroup[0]);
    levelGroup->removeChild(monsterGroup[1]);
    levelGroup->removeChild(monsterGroup[2]);
    levelGroup->removeChild(monsterGroup[3]);
    levelGroup->removeChild(monsterGroup[4]);
    levelGroup->addChild(monsterGroup[3]);
  }
  else if (num == 5) {
    if (bsps[4] == NULL)
      bsps[4] = new CaveQuakeBSP("maps/ptrip.bsp", *gammaCorrect);
    currBsp = bsps[4];
    pfNode *level5 = currBsp->level;
    levelDcs->addChild(level5);
    player->X = 53.0f;
    player->Y = -20.0f;
    player->Z = 0.0f;
    player->H = 270.0f;
    if (itemGroup[4] == NULL) {
      itemGroup[4] = new pfGroup; 
      itemGroup[4]->addChild(makeItem('h', 8.3f, -39.5f, 0.0f));
      itemGroup[4]->addChild(makeItem('h', 8.3f, -42.5f, 0.0f));
      itemGroup[4]->addChild(makeItem('h', 84.6f, -19.2f, -25.7f));
      itemGroup[4]->addChild(makeItem('h', 84.2f, -22.2f, -25.7f));
      itemGroup[4]->addChild(makeItem('h', -31.3f, -46.0f, -31.8f));
      itemGroup[4]->addChild(makeItem('h', -31.3f, -49.0f, -31.8f));
      itemGroup[4]->addChild(makeItem('h', 24.6f, -76.5f, -3.0f));
      itemGroup[4]->addChild(makeItem('a', 43.3f, 1.8f, 0.0f));
      itemGroup[4]->addChild(makeItem('a', 3.1f, -49.2f, -20.7f));
      itemGroup[4]->addChild(makeItem('a', -31.6f, 3.6f, -31.8f));
      itemGroup[4]->addChild(makeItem('a', -16.2f, -73.4f, -16.2f));
    }
    levelGroup->removeChild(itemGroup[0]);
    levelGroup->removeChild(itemGroup[1]);
    levelGroup->removeChild(itemGroup[2]);
    levelGroup->removeChild(itemGroup[3]);
    levelGroup->removeChild(itemGroup[4]);
    levelGroup->addChild(itemGroup[4]);
    if (monsterGroup[4] == NULL) {
      monsterGroup[4] = new pfGroup;
      monsterGroup[4]->addChild(makeMonster('s', 22.0f, -21.0f, -25.8f));
      //monsterGroup[4]->addChild(makeMonster('b', 33.2f, -11.3f, -25.7f));
      monsterGroup[4]->addChild(makeMonster('i', 33.2f, -11.3f, -25.7f));
      monsterGroup[4]->addChild(makeMonster('i', -17.2f, -65.1f, -16.3f));
      monsterGroup[4]->addChild(makeMonster('s', 23.3f, -97.7f, -25.8f));
      monsterGroup[4]->addChild(makeMonster('g', 14.0f, 11.4f, -0.1f));
    }
    levelGroup->removeChild(monsterGroup[0]);
    levelGroup->removeChild(monsterGroup[1]);
    levelGroup->removeChild(monsterGroup[2]);
    levelGroup->removeChild(monsterGroup[3]);
    levelGroup->removeChild(monsterGroup[4]);
    levelGroup->addChild(monsterGroup[4]);
  }
  else if (num == 6) {
    if (bsps[5] == NULL)
      bsps[5] = new CaveQuakeBSP("maps/custom.bsp", *gammaCorrect);
    currBsp = bsps[5];
    pfNode *level6 = currBsp->level;
    levelDcs->addChild(level6);
    player->X = playerStart[0]/10.0f;
    player->Y = playerStart[1]/10.0f; 
    player->Z = playerStart[2]/10.0f;
    player->H = playerStart[3];
  }

  *currLevel = num;
}

static void moveViper() {
  static float firstTime = pfGetTime();
  static float lastTime = pfGetTime();
  static int shipFlag = 1;
  float time = pfGetTime();

  float diff = time - firstTime - 5.0f;
  // target = -48.7 -70.7 -30.5
  if (diff < 28.0f) {
    if (diff > 4.0f && shipFlag) {
      shipFlag = 0;
      quakeSound("wShip", 1.0f);
    }
    viperDcs->setRot(120.0f - (diff-4.0)*3.0f, 0.0f, 0.0f);
    float v = (28.0f - diff)/3.0f;
    viperDcs->setTrans(-46.7f, -70.7f, -30.5f + v*v);
  } 

  lastTime = time;
}

static void checkForEvents() {
  static int checkCounter = 0;
  checkCounter++;

  if (checkCounter > 20) {
    checkCounter = 0;
    pfVec3 currPos(player->X, player->Y, player->Z);
    if (*currLevel == 1) {
      pfVec3 endPos(-174.4f, 153.7f, 0.0f); // was 9.9
      float distanceFromEnd = endPos.distance(currPos);
      if (distanceFromEnd < 7.0f)
        switchLevel(2);
    }
    else if (*currLevel == 2) {
      //pfVec3 elevatorPos(-9.8f, 141.1f, -19.7f);
      pfVec3 endPos(-90.7f, 6.1f, -42.0f);
      pfVec3 endPos2(8.0f, -79.1f, -1.7f);
      pfVec3 endPos3(87.9f, -177.3f, -0.05f);
      pfVec3 firstPos(88.0f, 229.9f, -18.4f); // was -25.4
      //float distanceFromEl = elevatorPos.distance(currPos);
      float distanceFromEnd = endPos.distance(currPos);
      float distanceFromEnd2 = endPos2.distance(currPos);
      float distanceFromEnd3 = endPos3.distance(currPos);
      float distanceFromFirst = firstPos.distance(currPos);
      if (distanceFromFirst < 7.0f)
        switchLevel(1, 1);
      //if (distanceFromEl < 5.0f)
        //player->Z = -0.2f;
      if (distanceFromEnd < 5.0f)
        switchLevel(3, 1);
      if (distanceFromEnd2 < 5.0f)
        switchLevel(3);
      if (distanceFromEnd3 < 5.0f) {
      // toggleScene->setVal(1);
        player->getStr()->setString("Not an exit!");
      }
    }
    else if (*currLevel == 3) {
      //pfVec3 elevatorPos(83.3f, 36.1f, -45.3f);
      //pfVec3 elevatorPos2(209.6f, 88.4f, -49.0f);
      pfVec3 endPos(246.9f, 89.4f, -32.9f);
      pfVec3 winPos(-53.8f, -34.5f, -29.1f);
      pfVec3 shipPos(-40.7f, -63.0f, -30.5f);
      //float distanceFromEl = elevatorPos.distance(currPos);
      //float distanceFromEl2 = elevatorPos2.distance(currPos);
      float distanceFromEnd = endPos.distance(currPos);
      float distanceFromWin = winPos.distance(currPos);
      float distanceFromShip = shipPos.distance(currPos);
      //if (distanceFromEl < 5.0f)
      //  player->Z = -25.3f;
      if (distanceFromEnd < 5.0f)
        switchLevel(2, 1);
      //if (distanceFromEl2 < 5.0f) {
      //  player->X = 208.5f;
      //  player->Y = 88.0f;
      //  player->Z = -33.1f;
      // }
      if (distanceFromWin < 5.0f) {
        static float lastT = 0.0f;
        if ((lastT == 0) || (pfGetTime() - lastT > 5.0f)) {
          quakeSound("wUplink", 1.0f);
          lastT = pfGetTime();
        }
        *viperToggle = 1;
        levelGroup->addChild(viperDcs);
      }
      else if (distanceFromWin < 40.0f) {
        static float lastTime = 0.0f;
        if ((lastTime == 0) || (pfGetTime() - lastTime > 10.5f)) {
          quakeSound("wComp", distanceFromWin);
          lastTime = pfGetTime();
        }
      }
      if (*viperToggle && distanceFromShip < 8.0f) {
        quakeSound("wWin", 1.0f);
        toggleScene->setVal(1);
      }
    } // if (*currLevel == 3)
    else if (*currLevel == 4) {
      //pfVec3 elevatorPos(8.2f, 43.2f, 15.8f);
      //float distanceFromEl = elevatorPos.distance(currPos); 
      //if (distanceFromEl < 6.0f)
      //  player->Z = 33.0f;
    }
    else if (*currLevel == 5) {
      //pfVec3 elevatorPos1(77.5f, 8.1f, -26.9f);
      //pfVec3 elevatorPos2(11.5f, 13.7f, -43.0f);
      //float distanceFromEl1 = elevatorPos1.distance(currPos);
      //float distanceFromEl2 = elevatorPos2.distance(currPos);
      //if (distanceFromEl1 < 5.0f)
      //  player->Z = 0.0f;
      //if (distanceFromEl2 < 5.0f)
      //  player->Z = 0.0f;
    }
  } // if (checkCounter > 20)
}

void printInfo() {

  cout << "\n\n";
  cout << "-----===== C A V E   Q U A K E   I I =====-----" << endl;
  cout << "               by Paul Rajlich\n" << endl;

  cout << "            prajlich@ncsa.uiuc.edu" << endl;
  cout << "http://brighton.ncsa.uiuc.edu/~prajlich/caveQuake\n" << endl;

  cout << "       pfQuake2 loaders by Rick Weyrauch" << endl;
  cout << "      Quake2 is a trademark of Id Software\n" << endl;

  cout << "To customize settings, copy " << getenv("QUAKE_HOME");
  cout << "/bin/run\nto your home directory and edit." << endl;
  cout << "\nEnjoy!\n" << endl;
  cout << "-----=====================================-----" << endl;

}

